/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.datastax.driver.examples.json;

import static com.datastax.driver.core.querybuilder.QueryBuilder.bindMarker;
import static com.datastax.driver.core.querybuilder.QueryBuilder.in;
import static com.datastax.driver.core.querybuilder.QueryBuilder.insertInto;
import static com.datastax.driver.core.querybuilder.QueryBuilder.select;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.CodecRegistry;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.TypeCodec;
import com.datastax.driver.extras.codecs.json.Jsr353JsonCodec;
import javax.json.Json;
import javax.json.JsonObject;
import javax.json.JsonStructure;

/**
 * Illustrates how to map a single table column of type {@code VARCHAR}, containing JSON payloads,
 * into a Java object using the <a href="https://jcp.org/en/jsr/detail?id=353">Java API for JSON
 * processing</a>.
 *
 * <p>This example makes usage of a custom {@link TypeCodec codec}, {@link Jsr353JsonCodec}, which
 * is declared in the driver-extras module. If you plan to follow this example, make sure to include
 * the following Maven dependencies in your project:
 *
 * <pre>{@code
 * <dependency>
 *     <groupId>org.apache.cassandra</groupId>
 *     <artifactId>cassandra-driver-extras</artifactId>
 *     <version>${driver.version}</version>
 * </dependency>
 *
 * <dependency>
 *     <groupId>javax.json</groupId>
 *     <artifactId>javax.json-api</artifactId>
 *     <version>${jsr353-api.version}</version>
 * </dependency>
 *
 * <dependency>
 *     <groupId>org.glassfish</groupId>
 *     <artifactId>javax.json</artifactId>
 *     <version>${jsr353-ri.version}</version>
 *     <scope>runtime</scope>
 * </dependency>
 * }</pre>
 *
 * This example also uses the {@link com.datastax.driver.core.querybuilder.QueryBuilder
 * QueryBuilder}; for examples using the "core" API, see {@link PlainTextJson} (they are easily
 * translatable to the queries in this class).
 *
 * <p>Preconditions: - a Cassandra cluster is running and accessible through the contacts points
 * identified by CONTACT_POINTS and PORT;
 *
 * <p>Side effects: - creates a new keyspace "examples" in the cluster. If a keyspace with this name
 * already exists, it will be reused; - creates a table "examples.json_jsr353_column". If it already
 * exists, it will be reused; - inserts data in the table.
 */
public class Jsr353JsonColumn {

  static String[] CONTACT_POINTS = {"127.0.0.1"};
  static int PORT = 9042;

  public static void main(String[] args) {
    Cluster cluster = null;
    try {

      // A codec to convert JSON payloads into JsonObject instances;
      // this codec is declared in the driver-extras module
      Jsr353JsonCodec userCodec = new Jsr353JsonCodec();

      cluster =
          Cluster.builder()
              .addContactPoints(CONTACT_POINTS)
              .withPort(PORT)
              .withCodecRegistry(new CodecRegistry().register(userCodec))
              .build();

      Session session = cluster.connect();

      createSchema(session);
      insertJsonColumn(session);
      selectJsonColumn(session);

    } finally {
      if (cluster != null) cluster.close();
    }
  }

  private static void createSchema(Session session) {
    session.execute(
        "CREATE KEYSPACE IF NOT EXISTS examples "
            + "WITH replication = {'class': 'SimpleStrategy', 'replication_factor': 1}");
    session.execute(
        "CREATE TABLE IF NOT EXISTS examples.json_jsr353_column("
            + "id int PRIMARY KEY, json text)");
  }

  // Mapping a JSON object to a table column
  private static void insertJsonColumn(Session session) {

    JsonObject alice = Json.createObjectBuilder().add("name", "alice").add("age", 30).build();

    JsonObject bob = Json.createObjectBuilder().add("name", "bob").add("age", 35).build();

    // Build and execute a simple statement
    Statement stmt =
        insertInto("examples", "json_jsr353_column")
            .value("id", 1)
            // the JSON object will be converted into a String and persisted into the VARCHAR column
            // "json"
            .value("json", alice);
    session.execute(stmt);

    // The JSON object can be a bound value if the statement is prepared
    // (we use a local variable here for the sake of example, but in a real application you would
    // cache and reuse
    // the prepared statement)
    PreparedStatement pst =
        session.prepare(
            insertInto("examples", "json_jsr353_column")
                .value("id", bindMarker("id"))
                .value("json", bindMarker("json")));
    session.execute(
        pst.bind()
            .setInt("id", 2)
            // note that the codec requires that the type passed to the set() method
            // be always JsonStructure, and not a subclass of it, such as JsonObject
            .set("json", bob, JsonStructure.class));
  }

  // Retrieving JSON objects from a table column
  private static void selectJsonColumn(Session session) {

    Statement stmt = select().from("examples", "json_jsr353_column").where(in("id", 1, 2));

    ResultSet rows = session.execute(stmt);

    for (Row row : rows) {
      int id = row.getInt("id");
      // retrieve the JSON payload and convert it to a JsonObject instance
      // note that the codec requires that the type passed to the get() method
      // be always JsonStructure, and not a subclass of it, such as JsonObject,
      // hence the need to downcast to JsonObject manually
      JsonObject user = (JsonObject) row.get("json", JsonStructure.class);
      // it is also possible to retrieve the raw JSON payload
      String json = row.getString("json");
      System.out.printf(
          "Retrieved row:%n" + "id           %d%n" + "user         %s%n" + "user (raw)   %s%n%n",
          id, user, json);
    }
  }
}
