/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import bk-shade.com.google.common.base.Charsets;
import bk-shade.com.google.common.collect.Lists;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.bookie.Cookie;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.util.HardLink;
import org.apache.bookkeeper.versioning.Version;
import org.apache.bookkeeper.versioning.Versioned;
import org.apache.bookkeeper.zookeeper.BoundExponentialBackoffRetryPolicy;
import org.apache.bookkeeper.zookeeper.ZooKeeperClient;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.io.FileUtils;
import org.apache.log4j.Appender;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Layout;
import org.apache.log4j.Level;
import org.apache.log4j.PatternLayout;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemUpgrade {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemUpgrade.class);
    static FilenameFilter bookieFilesFilter = new FilenameFilter(){

        private boolean containsIndexFiles(File dir, String name) {
            if (name.endsWith(".idx")) {
                return true;
            }
            try {
                String[] files;
                Long.parseLong(name, 16);
                File d = new File(dir, name);
                if (d.isDirectory() && (files = d.list()) != null) {
                    for (String f : files) {
                        if (!this.containsIndexFiles(d, f)) continue;
                        return true;
                    }
                }
            }
            catch (NumberFormatException nfe) {
                return false;
            }
            return false;
        }

        @Override
        public boolean accept(File dir, String name) {
            if (name.endsWith(".txn") || name.endsWith(".log") || name.equals("lastId") || name.equals("lastMark")) {
                return true;
            }
            return this.containsIndexFiles(dir, name);
        }
    };

    private static List<File> getAllDirectories(ServerConfiguration conf) {
        ArrayList<File> dirs = new ArrayList<File>();
        dirs.addAll(Lists.newArrayList(conf.getJournalDirs()));
        Collections.addAll(dirs, conf.getLedgerDirs());
        return dirs;
    }

    private static int detectPreviousVersion(File directory) throws IOException {
        String[] files = directory.list(bookieFilesFilter);
        File v2versionFile = new File(directory, "VERSION");
        if (!(files != null && files.length != 0 || v2versionFile.exists())) {
            return 4;
        }
        if (!v2versionFile.exists()) {
            return 1;
        }
        try (Scanner s = new Scanner(v2versionFile, Charsets.UTF_8.name());){
            int n = s.nextInt();
            return n;
        }
    }

    private static ZooKeeper newZookeeper(ServerConfiguration conf) throws BookieException.UpgradeException {
        try {
            int zkTimeout = conf.getZkTimeout();
            return ZooKeeperClient.newBuilder().connectString(conf.getZkServers()).sessionTimeoutMs(zkTimeout).operationRetryPolicy(new BoundExponentialBackoffRetryPolicy(zkTimeout, zkTimeout, Integer.MAX_VALUE)).build();
        }
        catch (InterruptedException ie) {
            throw new BookieException.UpgradeException(ie);
        }
        catch (IOException ioe) {
            throw new BookieException.UpgradeException(ioe);
        }
        catch (KeeperException ke) {
            throw new BookieException.UpgradeException(ke);
        }
    }

    private static void linkIndexDirectories(File srcPath, File targetPath) throws IOException {
        String[] files = srcPath.list();
        if (files == null) {
            return;
        }
        for (String f : files) {
            if (f.endsWith(".idx")) {
                if (!targetPath.mkdirs()) {
                    throw new IOException("Could not create target path [" + targetPath + "]");
                }
                HardLink.createHardLinkMult(srcPath, files, targetPath);
                return;
            }
            File newSrcPath = new File(srcPath, f);
            if (!newSrcPath.isDirectory()) continue;
            try {
                Long.parseLong(f, 16);
                FileSystemUpgrade.linkIndexDirectories(newSrcPath, new File(targetPath, f));
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
    }

    public static void upgrade(ServerConfiguration conf) throws BookieException.UpgradeException, InterruptedException {
        LOG.info("Upgrading...");
        try (ZooKeeper zk = FileSystemUpgrade.newZookeeper(conf);){
            HashMap<File, File> deferredMoves = new HashMap<File, File>();
            Cookie.Builder cookieBuilder = Cookie.generateCookie(conf);
            Cookie c = cookieBuilder.build();
            for (File file : FileSystemUpgrade.getAllDirectories(conf)) {
                LOG.info("Upgrading {}", (Object)file);
                int version = FileSystemUpgrade.detectPreviousVersion(file);
                if (version == 4) {
                    LOG.info("Directory is current, no need to upgrade");
                    continue;
                }
                try {
                    File curDir = new File(file, "current");
                    File tmpDir = new File(file, "upgradeTmp." + System.nanoTime());
                    deferredMoves.put(curDir, tmpDir);
                    if (!tmpDir.mkdirs()) {
                        throw new BookieException.UpgradeException("Could not create temporary directory " + tmpDir);
                    }
                    c.writeToDirectory(tmpDir);
                    String[] files = file.list(new FilenameFilter(){

                        @Override
                        public boolean accept(File dir, String name) {
                            return bookieFilesFilter.accept(dir, name) && !new File(dir, name).isDirectory();
                        }
                    });
                    HardLink.createHardLinkMult(file, files, tmpDir);
                    FileSystemUpgrade.linkIndexDirectories(file, tmpDir);
                }
                catch (IOException ioe) {
                    LOG.error("Error upgrading {}", (Object)file);
                    throw new BookieException.UpgradeException(ioe);
                }
            }
            for (Map.Entry entry : deferredMoves.entrySet()) {
                try {
                    FileUtils.moveDirectory((File)((File)entry.getValue()), (File)((File)entry.getKey()));
                }
                catch (IOException ioe) {
                    String err = String.format("Error moving upgraded directories into place %s -> %s ", entry.getValue(), entry.getKey());
                    LOG.error(err, (Throwable)ioe);
                    throw new BookieException.UpgradeException(ioe);
                }
            }
            if (deferredMoves.isEmpty()) {
                return;
            }
            try {
                c.writeToZooKeeper(zk, conf, Version.NEW);
            }
            catch (KeeperException ke) {
                LOG.error("Error writing cookie to zookeeper");
                throw new BookieException.UpgradeException(ke);
            }
        }
        LOG.info("Done");
    }

    public static void finalizeUpgrade(ServerConfiguration conf) throws BookieException.UpgradeException, InterruptedException {
        LOG.info("Finalizing upgrade...");
        for (File d : FileSystemUpgrade.getAllDirectories(conf)) {
            LOG.info("Finalizing {}", (Object)d);
            try {
                File[] files;
                File v2versionFile;
                int version = FileSystemUpgrade.detectPreviousVersion(d);
                if (version >= 3) continue;
                if (version == 2 && !(v2versionFile = new File(d, "VERSION")).delete()) {
                    LOG.warn("Could not delete old version file {}", (Object)v2versionFile);
                }
                if ((files = d.listFiles(bookieFilesFilter)) == null) continue;
                for (File f : files) {
                    if (f.isDirectory()) {
                        FileUtils.deleteDirectory((File)f);
                        continue;
                    }
                    if (f.delete()) continue;
                    LOG.warn("Could not delete {}", (Object)f);
                }
            }
            catch (IOException ioe) {
                LOG.error("Error finalizing {}", (Object)d);
                throw new BookieException.UpgradeException(ioe);
            }
        }
        LOG.info("Done");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void rollback(ServerConfiguration conf) throws BookieException.UpgradeException, InterruptedException {
        LOG.info("Rolling back upgrade...");
        try (ZooKeeper zk = FileSystemUpgrade.newZookeeper(conf);){
            for (File d : FileSystemUpgrade.getAllDirectories(conf)) {
                LOG.info("Rolling back {}", (Object)d);
                try {
                    int version = FileSystemUpgrade.detectPreviousVersion(d);
                    if (version <= 4) {
                        File curDir = new File(d, "current");
                        FileUtils.deleteDirectory((File)curDir);
                        continue;
                    }
                    throw new BookieException.UpgradeException("Cannot rollback as previous data does not exist");
                }
                catch (IOException ioe) {
                    LOG.error("Error rolling back {}", (Object)d);
                    throw new BookieException.UpgradeException(ioe);
                }
            }
            try {
                Versioned<Cookie> cookie = Cookie.readFromZooKeeper(zk, conf);
                cookie.getValue().deleteFromZooKeeper(zk, conf, cookie.getVersion());
            }
            catch (KeeperException ke) {
                LOG.error("Error deleting cookie from ZooKeeper");
                throw new BookieException.UpgradeException(ke);
            }
            catch (IOException ioe) {
                LOG.error("I/O Error deleting cookie from ZooKeeper");
                throw new BookieException.UpgradeException(ioe);
            }
        }
        LOG.info("Done");
    }

    private static void printHelp(Options opts) {
        HelpFormatter hf = new HelpFormatter();
        hf.printHelp("FileSystemUpgrade [options]", opts);
    }

    public static void main(String[] args) throws Exception {
        org.apache.log4j.Logger root = org.apache.log4j.Logger.getRootLogger();
        root.addAppender((Appender)new ConsoleAppender((Layout)new PatternLayout("%-5p [%t]: %m%n")));
        root.setLevel(Level.ERROR);
        org.apache.log4j.Logger.getLogger(FileSystemUpgrade.class).setLevel(Level.INFO);
        Options opts = new Options();
        opts.addOption("c", "conf", true, "Configuration for Bookie");
        opts.addOption("u", "upgrade", false, "Upgrade bookie directories");
        opts.addOption("f", "finalize", false, "Finalize upgrade");
        opts.addOption("r", "rollback", false, "Rollback upgrade");
        opts.addOption("h", "help", false, "Print help message");
        BasicParser parser = new BasicParser();
        CommandLine cmdLine = parser.parse(opts, args);
        if (cmdLine.hasOption("h")) {
            FileSystemUpgrade.printHelp(opts);
            return;
        }
        if (!cmdLine.hasOption("c")) {
            String err = "Cannot upgrade without configuration";
            LOG.error(err);
            FileSystemUpgrade.printHelp(opts);
            throw new IllegalArgumentException(err);
        }
        String confFile = cmdLine.getOptionValue("c");
        ServerConfiguration conf = new ServerConfiguration();
        try {
            conf.loadConf(new File(confFile).toURI().toURL());
        }
        catch (MalformedURLException mue) {
            LOG.error("Could not open configuration file " + confFile, (Throwable)mue);
            throw new IllegalArgumentException();
        }
        catch (ConfigurationException ce) {
            LOG.error("Invalid configuration file " + confFile, (Throwable)ce);
            throw new IllegalArgumentException();
        }
        if (cmdLine.hasOption("u")) {
            FileSystemUpgrade.upgrade(conf);
        } else if (cmdLine.hasOption("r")) {
            FileSystemUpgrade.rollback(conf);
        } else if (cmdLine.hasOption("f")) {
            FileSystemUpgrade.finalizeUpgrade(conf);
        } else {
            String err = "Must specify -upgrade, -finalize or -rollback";
            LOG.error(err);
            FileSystemUpgrade.printHelp(opts);
            throw new IllegalArgumentException(err);
        }
    }
}

