/*
 * Decompiled with CFR 0.152.
 */
package org.apache.parquet.column.values.bloomfilter;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.Objects;
import java.util.TreeSet;
import org.apache.parquet.Preconditions;
import org.apache.parquet.column.ColumnDescriptor;
import org.apache.parquet.column.values.bloomfilter.BlockSplitBloomFilter;
import org.apache.parquet.column.values.bloomfilter.BloomFilter;
import org.apache.parquet.io.api.Binary;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DynamicBlockBloomFilter
implements BloomFilter {
    private static final Logger LOG = LoggerFactory.getLogger(DynamicBlockBloomFilter.class);
    private final TreeSet<BloomFilterCandidate> candidates = new TreeSet();
    private BloomFilterCandidate maxCandidate;
    private int distinctValueCounter = 0;
    private boolean finalized = false;
    private int maximumBytes = 0x8000000;
    private int minimumBytes = 32;
    private final BloomFilter.HashStrategy hashStrategy;
    private ColumnDescriptor column;

    public DynamicBlockBloomFilter(int numBytes, int candidatesNum, double fpp, ColumnDescriptor column) {
        this(numBytes, 32, 0x8000000, BloomFilter.HashStrategy.XXH64, fpp, candidatesNum, column);
    }

    public DynamicBlockBloomFilter(int numBytes, int maximumBytes, int candidatesNum, double fpp, ColumnDescriptor column) {
        this(numBytes, 32, maximumBytes, BloomFilter.HashStrategy.XXH64, fpp, candidatesNum, column);
    }

    public DynamicBlockBloomFilter(int numBytes, int minimumBytes, int maximumBytes, BloomFilter.HashStrategy hashStrategy, double fpp, int candidatesNum, ColumnDescriptor column) {
        if (minimumBytes > maximumBytes) {
            throw new IllegalArgumentException("the minimum bytes should be less or equal than maximum bytes");
        }
        if (minimumBytes > 32 && minimumBytes < 0x8000000) {
            this.minimumBytes = minimumBytes;
        }
        if (maximumBytes > 32 && maximumBytes < 0x8000000) {
            this.maximumBytes = maximumBytes;
        }
        this.column = column;
        switch (hashStrategy) {
            case XXH64: {
                this.hashStrategy = hashStrategy;
                break;
            }
            default: {
                throw new RuntimeException("Unsupported hash strategy");
            }
        }
        this.initCandidates(numBytes, candidatesNum, fpp);
    }

    private void initCandidates(int maxBytes, int candidatesNum, double fpp) {
        int candidateExpectedNDV;
        int candidateByteSize = this.calculateTwoPowerSize(maxBytes);
        for (int i = 1; i <= candidatesNum && (candidateExpectedNDV = this.expectedNDV(candidateByteSize, fpp)) > 0; ++i) {
            BloomFilterCandidate candidate = new BloomFilterCandidate(candidateExpectedNDV, candidateByteSize, this.minimumBytes, this.maximumBytes, this.hashStrategy);
            this.candidates.add(candidate);
            candidateByteSize = this.calculateTwoPowerSize(candidateByteSize / 2);
        }
        this.maxCandidate = this.candidates.last();
    }

    private int expectedNDV(int numBytes, double fpp) {
        int expectedNDV = 0;
        int optimalBytes = 0;
        int step = 500;
        while (optimalBytes < numBytes) {
            optimalBytes = BlockSplitBloomFilter.optimalNumOfBits(expectedNDV += step, fpp);
        }
        if ((expectedNDV -= step) <= 0 && numBytes >= 256) {
            expectedNDV = 1;
        } else if (expectedNDV <= 0) {
            expectedNDV = 0;
        }
        return expectedNDV;
    }

    private int calculateTwoPowerSize(int numBytes) {
        if (numBytes < this.minimumBytes) {
            numBytes = this.minimumBytes;
        }
        if ((numBytes & numBytes - 1) != 0) {
            numBytes = Integer.highestOneBit(numBytes) >> 1;
        }
        if (numBytes > this.maximumBytes || numBytes < 0) {
            numBytes = this.maximumBytes;
        }
        if (numBytes < this.minimumBytes) {
            numBytes = this.minimumBytes;
        }
        return numBytes;
    }

    protected BloomFilterCandidate optimalCandidate() {
        return (BloomFilterCandidate)this.candidates.stream().min(BloomFilterCandidate::compareTo).get();
    }

    protected TreeSet<BloomFilterCandidate> getCandidates() {
        return this.candidates;
    }

    @Override
    public void writeTo(OutputStream out) throws IOException {
        this.finalized = true;
        BloomFilterCandidate optimalBloomFilter = this.optimalCandidate();
        optimalBloomFilter.bloomFilter.writeTo(out);
        String columnName = this.column != null && this.column.getPath() != null ? Arrays.toString(this.column.getPath()) : "unknown";
        LOG.info("The number of distinct values in {} is approximately {}, the optimal bloom filter NDV is {}, byte size is {}.", new Object[]{columnName, this.distinctValueCounter, optimalBloomFilter.getExpectedNDV(), optimalBloomFilter.bloomFilter.getBitsetSize()});
    }

    @Override
    public void insertHash(long hash) {
        Preconditions.checkArgument((!this.finalized ? 1 : 0) != 0, (String)"Dynamic bloom filter insertion has been mark as finalized, no more data is allowed!");
        if (!this.maxCandidate.bloomFilter.findHash(hash)) {
            ++this.distinctValueCounter;
        }
        this.candidates.removeIf(candidate -> candidate.getExpectedNDV() < this.distinctValueCounter && candidate != this.maxCandidate);
        this.candidates.forEach(candidate -> candidate.getBloomFilter().insertHash(hash));
    }

    @Override
    public int getBitsetSize() {
        return this.optimalCandidate().getBloomFilter().getBitsetSize();
    }

    @Override
    public boolean findHash(long hash) {
        return this.maxCandidate.bloomFilter.findHash(hash);
    }

    @Override
    public long hash(Object value) {
        return this.maxCandidate.bloomFilter.hash(value);
    }

    @Override
    public BloomFilter.HashStrategy getHashStrategy() {
        return this.maxCandidate.bloomFilter.getHashStrategy();
    }

    @Override
    public BloomFilter.Algorithm getAlgorithm() {
        return this.maxCandidate.bloomFilter.getAlgorithm();
    }

    @Override
    public BloomFilter.Compression getCompression() {
        return this.maxCandidate.bloomFilter.getCompression();
    }

    @Override
    public long hash(int value) {
        return this.maxCandidate.bloomFilter.hash(value);
    }

    @Override
    public long hash(long value) {
        return this.maxCandidate.bloomFilter.hash(value);
    }

    @Override
    public long hash(double value) {
        return this.maxCandidate.bloomFilter.hash(value);
    }

    @Override
    public long hash(float value) {
        return this.maxCandidate.bloomFilter.hash(value);
    }

    @Override
    public long hash(Binary value) {
        return this.maxCandidate.bloomFilter.hash(value);
    }

    protected class BloomFilterCandidate
    implements Comparable<BloomFilterCandidate> {
        private BlockSplitBloomFilter bloomFilter;
        private int expectedNDV;

        public BloomFilterCandidate(int expectedNDV, int candidateBytes, int minimumBytes, int maximumBytes, BloomFilter.HashStrategy hashStrategy) {
            this.bloomFilter = new BlockSplitBloomFilter(candidateBytes, minimumBytes, maximumBytes, hashStrategy);
            this.expectedNDV = expectedNDV;
        }

        public BlockSplitBloomFilter getBloomFilter() {
            return this.bloomFilter;
        }

        public void setBloomFilter(BlockSplitBloomFilter bloomFilter) {
            this.bloomFilter = bloomFilter;
        }

        public int getExpectedNDV() {
            return this.expectedNDV;
        }

        public void setExpectedNDV(int expectedNDV) {
            this.expectedNDV = expectedNDV;
        }

        @Override
        public int compareTo(BloomFilterCandidate o) {
            return this.bloomFilter.getBitsetSize() - o.bloomFilter.getBitsetSize();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            BloomFilterCandidate that = (BloomFilterCandidate)o;
            return this.expectedNDV == that.expectedNDV && Objects.equals(this.bloomFilter, that.bloomFilter);
        }

        public int hashCode() {
            return Objects.hash(this.bloomFilter, this.expectedNDV);
        }
    }
}

