/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.server.coordinator.balancer;

import com.google.common.base.Preconditions;
import com.google.common.collect.Ordering;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.ListIterator;
import java.util.NavigableMap;
import java.util.NavigableSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.math3.util.FastMath;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.Intervals;
import org.apache.druid.java.util.common.granularity.DurationGranularity;
import org.apache.druid.java.util.common.guava.Comparators;
import org.apache.druid.server.coordinator.balancer.CostBalancerStrategy;
import org.apache.druid.timeline.DataSegment;
import org.joda.time.Interval;
import org.joda.time.ReadableInstant;
import org.joda.time.ReadableInterval;

public class SegmentsCostCache {
    private static final double HALF_LIFE_DAYS = 1.0;
    private static final double LAMBDA = Math.log(2.0) / 1.0;
    private static final double MILLIS_FACTOR = (double)TimeUnit.DAYS.toMillis(1L) / LAMBDA;
    private static final long LIFE_THRESHOLD = TimeUnit.DAYS.toMillis(30L);
    private static final long BUCKET_INTERVAL = TimeUnit.DAYS.toMillis(15L);
    private static final DurationGranularity BUCKET_GRANULARITY = new DurationGranularity(BUCKET_INTERVAL, 0L);
    private static final Comparator<DataSegment> SEGMENT_INTERVAL_COMPARATOR = Comparator.comparing(DataSegment::getInterval, Comparators.intervalsByStartThenEnd());
    private static final Comparator<Bucket> BUCKET_INTERVAL_COMPARATOR = Comparator.comparing(Bucket::getInterval, Comparators.intervalsByStartThenEnd());
    private static final Ordering<DataSegment> SEGMENT_ORDERING = Ordering.from(SEGMENT_INTERVAL_COMPARATOR);
    private static final Ordering<Bucket> BUCKET_ORDERING = Ordering.from(BUCKET_INTERVAL_COMPARATOR);
    private final ArrayList<Bucket> sortedBuckets;
    private final ArrayList<Interval> intervals;

    SegmentsCostCache(ArrayList<Bucket> sortedBuckets) {
        this.sortedBuckets = (ArrayList)Preconditions.checkNotNull(sortedBuckets, (Object)"buckets should not be null");
        this.intervals = sortedBuckets.stream().map(Bucket::getInterval).collect(Collectors.toCollection(ArrayList::new));
        Preconditions.checkArgument((boolean)BUCKET_ORDERING.isOrdered(sortedBuckets), (Object)"buckets must be ordered by interval");
    }

    public double cost(DataSegment segment) {
        Bucket bucket;
        double cost = 0.0;
        int index = Collections.binarySearch(this.intervals, segment.getInterval(), Comparators.intervalsByStartThenEnd());
        index = index >= 0 ? index : -index - 1;
        ListIterator<Bucket> it = this.sortedBuckets.listIterator(index);
        while (it.hasNext() && (bucket = it.next()).inCalculationInterval(segment)) {
            cost += bucket.cost(segment);
        }
        it = this.sortedBuckets.listIterator(index);
        while (it.hasPrevious() && (bucket = it.previous()).inCalculationInterval(segment)) {
            cost += bucket.cost(segment);
        }
        return cost;
    }

    public static Builder builder() {
        return new Builder();
    }

    static class SegmentAndSum
    implements Comparable<SegmentAndSum> {
        private final DataSegment dataSegment;
        private double leftSum;
        private double rightSum;

        SegmentAndSum(DataSegment dataSegment, double leftSum, double rightSum) {
            this.dataSegment = dataSegment;
            this.leftSum = leftSum;
            this.rightSum = rightSum;
        }

        @Override
        public int compareTo(SegmentAndSum o) {
            int c = Comparators.intervalsByStartThenEnd().compare(this.dataSegment.getInterval(), o.dataSegment.getInterval());
            return c != 0 ? c : this.dataSegment.compareTo(o.dataSegment);
        }

        public boolean equals(Object obj) {
            throw new UnsupportedOperationException("Use SegmentAndSum.compareTo()");
        }

        public int hashCode() {
            throw new UnsupportedOperationException();
        }
    }

    static class Bucket {
        private final Interval interval;
        private final Interval calculationInterval;
        private final ArrayList<DataSegment> sortedSegments;
        private final double[] leftSum;
        private final double[] rightSum;

        Bucket(Interval interval, ArrayList<DataSegment> sortedSegments, double[] leftSum, double[] rightSum) {
            this.interval = (Interval)Preconditions.checkNotNull((Object)interval, (Object)"interval");
            this.sortedSegments = (ArrayList)Preconditions.checkNotNull(sortedSegments, (Object)"sortedSegments");
            this.leftSum = (double[])Preconditions.checkNotNull((Object)leftSum, (Object)"leftSum");
            this.rightSum = (double[])Preconditions.checkNotNull((Object)rightSum, (Object)"rightSum");
            Preconditions.checkArgument((sortedSegments.size() == leftSum.length && sortedSegments.size() == rightSum.length ? 1 : 0) != 0);
            Preconditions.checkArgument((boolean)SEGMENT_ORDERING.isOrdered(sortedSegments));
            this.calculationInterval = new Interval((ReadableInstant)interval.getStart().minus(LIFE_THRESHOLD), (ReadableInstant)interval.getEnd().plus(LIFE_THRESHOLD));
        }

        Interval getInterval() {
            return this.interval;
        }

        boolean inCalculationInterval(DataSegment dataSegment) {
            return this.calculationInterval.overlaps((ReadableInterval)dataSegment.getInterval());
        }

        double cost(DataSegment dataSegment) {
            double t0 = Bucket.convertStart(dataSegment, this.interval);
            double t1 = Bucket.convertEnd(dataSegment, this.interval);
            if (!this.inCalculationInterval(dataSegment)) {
                throw new ISE("Segment is not within calculation interval", new Object[0]);
            }
            int index = Collections.binarySearch(this.sortedSegments, dataSegment, SEGMENT_INTERVAL_COMPARATOR);
            index = index >= 0 ? index : -index - 1;
            return this.addLeftCost(dataSegment, t0, t1, index) + this.rightCost(dataSegment, t0, t1, index);
        }

        private double addLeftCost(DataSegment dataSegment, double t0, double t1, int index) {
            int leftIndex;
            double leftCost = 0.0;
            for (leftIndex = index - 1; leftIndex >= 0 && this.sortedSegments.get(leftIndex).getInterval().overlaps((ReadableInterval)dataSegment.getInterval()); --leftIndex) {
                double start = Bucket.convertStart(this.sortedSegments.get(leftIndex), this.interval);
                double end = Bucket.convertEnd(this.sortedSegments.get(leftIndex), this.interval);
                leftCost += CostBalancerStrategy.intervalCost(end - start, t0 - start, t1 - start);
            }
            if (leftIndex >= 0) {
                leftCost += this.leftSum[leftIndex] * (FastMath.exp((double)(-t1)) - FastMath.exp((double)(-t0)));
            }
            return leftCost;
        }

        private double rightCost(DataSegment dataSegment, double t0, double t1, int index) {
            int rightIndex;
            double rightCost = 0.0;
            for (rightIndex = index; rightIndex < this.sortedSegments.size() && this.sortedSegments.get(rightIndex).getInterval().overlaps((ReadableInterval)dataSegment.getInterval()); ++rightIndex) {
                double start = Bucket.convertStart(this.sortedSegments.get(rightIndex), this.interval);
                double end = Bucket.convertEnd(this.sortedSegments.get(rightIndex), this.interval);
                rightCost += CostBalancerStrategy.intervalCost(t1 - t0, start - t0, end - t0);
            }
            if (rightIndex < this.sortedSegments.size()) {
                rightCost += this.rightSum[rightIndex] * (FastMath.exp((double)t0) - FastMath.exp((double)t1));
            }
            return rightCost;
        }

        private static double convertStart(DataSegment dataSegment, Interval interval) {
            return Bucket.toLocalInterval(dataSegment.getInterval().getStartMillis(), interval);
        }

        private static double convertEnd(DataSegment dataSegment, Interval interval) {
            return Bucket.toLocalInterval(dataSegment.getInterval().getEndMillis(), interval);
        }

        private static double toLocalInterval(long millis, Interval interval) {
            return (double)millis / MILLIS_FACTOR - (double)interval.getStartMillis() / MILLIS_FACTOR;
        }

        public static Builder builder(Interval interval) {
            return new Builder(interval);
        }

        static class Builder {
            private final Interval interval;
            private final NavigableSet<SegmentAndSum> segments = new TreeSet<SegmentAndSum>();

            public Builder(Interval interval) {
                this.interval = interval;
            }

            public Builder addSegment(DataSegment dataSegment) {
                SegmentAndSum higher;
                if (!this.interval.contains(dataSegment.getInterval().getStartMillis())) {
                    throw new ISE("Failed to add segment to bucket: interval is not covered by this bucket", new Object[0]);
                }
                double t0 = Bucket.convertStart(dataSegment, this.interval);
                double t1 = Bucket.convertEnd(dataSegment, this.interval);
                double leftValue = FastMath.exp((double)t0) - FastMath.exp((double)t1);
                double rightValue = FastMath.exp((double)(-t1)) - FastMath.exp((double)(-t0));
                SegmentAndSum segmentAndSum = new SegmentAndSum(dataSegment, leftValue, rightValue);
                this.segments.tailSet(segmentAndSum).forEach(v -> v.leftSum += leftValue);
                this.segments.headSet(segmentAndSum).forEach(v -> v.rightSum += rightValue);
                SegmentAndSum lower = this.segments.lower(segmentAndSum);
                if (lower != null) {
                    segmentAndSum.leftSum = leftValue + lower.leftSum;
                }
                if ((higher = this.segments.higher(segmentAndSum)) != null) {
                    segmentAndSum.rightSum = rightValue + higher.rightSum;
                }
                if (!this.segments.add(segmentAndSum)) {
                    throw new ISE("expect new segment", new Object[0]);
                }
                return this;
            }

            public Builder removeSegment(DataSegment dataSegment) {
                SegmentAndSum segmentAndSum = new SegmentAndSum(dataSegment, 0.0, 0.0);
                if (!this.segments.remove(segmentAndSum)) {
                    return this;
                }
                double t0 = Bucket.convertStart(dataSegment, this.interval);
                double t1 = Bucket.convertEnd(dataSegment, this.interval);
                double leftValue = FastMath.exp((double)t0) - FastMath.exp((double)t1);
                double rightValue = FastMath.exp((double)(-t1)) - FastMath.exp((double)(-t0));
                this.segments.tailSet(segmentAndSum).forEach(v -> v.leftSum -= leftValue);
                this.segments.headSet(segmentAndSum).forEach(v -> v.rightSum -= rightValue);
                return this;
            }

            public boolean isEmpty() {
                return this.segments.isEmpty();
            }

            public Bucket build() {
                ArrayList<DataSegment> segmentsList = new ArrayList<DataSegment>(this.segments.size());
                double[] leftSum = new double[this.segments.size()];
                double[] rightSum = new double[this.segments.size()];
                int i = 0;
                for (SegmentAndSum segmentAndSum : this.segments) {
                    segmentsList.add(segmentAndSum.dataSegment);
                    leftSum[i] = segmentAndSum.leftSum;
                    rightSum[i] = segmentAndSum.rightSum;
                    ++i;
                }
                long bucketEndMillis = segmentsList.stream().mapToLong(s -> s.getInterval().getEndMillis()).max().orElseGet(() -> ((Interval)this.interval).getEndMillis());
                return new Bucket(Intervals.utc((long)this.interval.getStartMillis(), (long)bucketEndMillis), segmentsList, leftSum, rightSum);
            }
        }
    }

    public static class Builder {
        private NavigableMap<Interval, Bucket.Builder> buckets = new TreeMap<Interval, Bucket.Builder>(Comparators.intervalsByStartThenEnd());

        public Builder addSegment(DataSegment segment) {
            Bucket.Builder builder = this.buckets.computeIfAbsent(Builder.getBucketInterval(segment), Bucket::builder);
            builder.addSegment(segment);
            return this;
        }

        public Builder removeSegment(DataSegment segment) {
            Interval interval = Builder.getBucketInterval(segment);
            this.buckets.computeIfPresent(interval, (i, builder) -> builder.removeSegment(segment).isEmpty() ? null : builder);
            return this;
        }

        public boolean isEmpty() {
            return this.buckets.isEmpty();
        }

        public SegmentsCostCache build() {
            return new SegmentsCostCache(this.buckets.entrySet().stream().map(entry -> ((Bucket.Builder)entry.getValue()).build()).collect(Collectors.toCollection(ArrayList::new)));
        }

        private static Interval getBucketInterval(DataSegment segment) {
            return BUCKET_GRANULARITY.bucket(segment.getInterval().getStart());
        }
    }
}

