/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.segment.loading;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.inject.Inject;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.output.NullOutputStream;
import org.apache.druid.error.DruidException;
import org.apache.druid.guice.annotations.Json;
import org.apache.druid.java.util.common.FileUtils;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.concurrent.Execs;
import org.apache.druid.java.util.emitter.EmittingLogger;
import org.apache.druid.segment.IndexIO;
import org.apache.druid.segment.ReferenceCountingSegment;
import org.apache.druid.segment.Segment;
import org.apache.druid.segment.SegmentLazyLoadFailCallback;
import org.apache.druid.segment.loading.DataSegmentPusher;
import org.apache.druid.segment.loading.LoadSpec;
import org.apache.druid.segment.loading.MMappedQueryableSegmentizerFactory;
import org.apache.druid.segment.loading.SegmentCacheManager;
import org.apache.druid.segment.loading.SegmentLoaderConfig;
import org.apache.druid.segment.loading.SegmentLoadingException;
import org.apache.druid.segment.loading.SegmentizerFactory;
import org.apache.druid.segment.loading.StorageLocation;
import org.apache.druid.segment.loading.StorageLocationSelectorStrategy;
import org.apache.druid.timeline.DataSegment;
import org.apache.druid.timeline.SegmentId;
import org.apache.druid.timeline.partition.ShardSpec;

public class SegmentLocalCacheManager
implements SegmentCacheManager {
    @VisibleForTesting
    static final String DOWNLOAD_START_MARKER_FILE_NAME = "downloadStartMarker";
    private static final EmittingLogger log = new EmittingLogger(SegmentLocalCacheManager.class);
    private final SegmentLoaderConfig config;
    private final ObjectMapper jsonMapper;
    private final List<StorageLocation> locations;
    private final Object directoryWriteRemoveLock = new Object();
    private final ConcurrentHashMap<DataSegment, ReferenceCountingLock> segmentLocks = new ConcurrentHashMap();
    private final StorageLocationSelectorStrategy strategy;
    private final IndexIO indexIO;
    private ExecutorService loadOnBootstrapExec = null;
    private ExecutorService loadOnDownloadExec = null;

    @Inject
    public SegmentLocalCacheManager(List<StorageLocation> locations, SegmentLoaderConfig config, @Nonnull StorageLocationSelectorStrategy strategy, IndexIO indexIO, @Json ObjectMapper mapper) {
        this.config = config;
        this.jsonMapper = mapper;
        this.locations = locations;
        this.strategy = strategy;
        this.indexIO = indexIO;
        log.info("Using storage location strategy[%s].", new Object[]{this.strategy.getClass().getSimpleName()});
        log.info("Number of threads to load segments into page cache - on bootstrap: [%d], on download: [%d].", new Object[]{config.getNumThreadsToLoadSegmentsIntoPageCacheOnBootstrap(), config.getNumThreadsToLoadSegmentsIntoPageCacheOnDownload()});
        if (config.getNumThreadsToLoadSegmentsIntoPageCacheOnBootstrap() > 0) {
            this.loadOnBootstrapExec = Execs.multiThreaded((int)config.getNumThreadsToLoadSegmentsIntoPageCacheOnBootstrap(), (String)"Load-SegmentsIntoPageCacheOnBootstrap-%s");
        }
        if (config.getNumThreadsToLoadSegmentsIntoPageCacheOnDownload() > 0) {
            this.loadOnDownloadExec = Executors.newFixedThreadPool(config.getNumThreadsToLoadSegmentsIntoPageCacheOnDownload(), Execs.makeThreadFactory((String)"LoadSegmentsIntoPageCacheOnDownload-%s"));
        }
    }

    @Override
    public boolean canHandleSegments() {
        boolean isLocationsValid = this.locations != null && !this.locations.isEmpty();
        boolean isLocationsConfigValid = this.config.getLocations() != null && !this.config.getLocations().isEmpty();
        return isLocationsValid || isLocationsConfigValid;
    }

    @Override
    public List<DataSegment> getCachedSegments() throws IOException {
        if (!this.canHandleSegments()) {
            throw DruidException.defensive((String)"canHandleSegments() is false. getCachedSegments() must be invoked only when canHandleSegments() returns true.", (Object[])new Object[0]);
        }
        File infoDir = this.getEffectiveInfoDir();
        FileUtils.mkdirp((File)infoDir);
        ArrayList<DataSegment> cachedSegments = new ArrayList<DataSegment>();
        File[] segmentsToLoad = infoDir.listFiles();
        int ignored = 0;
        for (int i = 0; i < segmentsToLoad.length; ++i) {
            File file = segmentsToLoad[i];
            log.info("Loading segment cache file [%d/%d][%s].", new Object[]{i + 1, segmentsToLoad.length, file});
            try {
                DataSegment segment = (DataSegment)this.jsonMapper.readValue(file, DataSegment.class);
                if (!segment.getId().toString().equals(file.getName())) {
                    log.warn("Ignoring cache file[%s] for segment[%s].", new Object[]{file.getPath(), segment.getId()});
                    ++ignored;
                    continue;
                }
                if (this.isSegmentCached(segment)) {
                    cachedSegments.add(segment);
                    continue;
                }
                SegmentId segmentId = segment.getId();
                log.warn("Unable to find cache file for segment[%s]. Deleting lookup entry.", new Object[]{segmentId});
                this.removeInfoFile(segment);
                continue;
            }
            catch (Exception e) {
                log.makeAlert((Throwable)e, "Failed to load segment from segment cache file.", new Object[0]).addData("file", (Object)file).emit();
            }
        }
        if (ignored > 0) {
            log.makeAlert("Ignored misnamed segment cache files on startup.", new Object[0]).addData("numIgnored", (Object)ignored).emit();
        }
        return cachedSegments;
    }

    @Override
    public void storeInfoFile(DataSegment segment) throws IOException {
        File segmentInfoCacheFile = new File(this.getEffectiveInfoDir(), segment.getId().toString());
        if (!segmentInfoCacheFile.exists()) {
            this.jsonMapper.writeValue(segmentInfoCacheFile, (Object)segment);
        }
    }

    @Override
    public void removeInfoFile(DataSegment segment) {
        File segmentInfoCacheFile = new File(this.getEffectiveInfoDir(), segment.getId().toString());
        if (!segmentInfoCacheFile.delete()) {
            log.warn("Unable to delete cache file[%s] for segment[%s].", new Object[]{segmentInfoCacheFile, segment.getId()});
        }
    }

    @Override
    public ReferenceCountingSegment getSegment(DataSegment dataSegment) throws SegmentLoadingException {
        File segmentFiles = this.getSegmentFiles(dataSegment);
        SegmentizerFactory factory = this.getSegmentFactory(segmentFiles);
        Segment segment = factory.factorize(dataSegment, segmentFiles, false, SegmentLazyLoadFailCallback.NOOP);
        return ReferenceCountingSegment.wrapSegment((Segment)segment, (ShardSpec)dataSegment.getShardSpec());
    }

    @Override
    public ReferenceCountingSegment getBootstrapSegment(DataSegment dataSegment, SegmentLazyLoadFailCallback loadFailed) throws SegmentLoadingException {
        File segmentFiles = this.getSegmentFiles(dataSegment);
        SegmentizerFactory factory = this.getSegmentFactory(segmentFiles);
        Segment segment = factory.factorize(dataSegment, segmentFiles, this.config.isLazyLoadOnStart(), loadFailed);
        return ReferenceCountingSegment.wrapSegment((Segment)segment, (ShardSpec)dataSegment.getShardSpec());
    }

    private SegmentizerFactory getSegmentFactory(File segmentFiles) throws SegmentLoadingException {
        MMappedQueryableSegmentizerFactory factory;
        File factoryJson = new File(segmentFiles, "factory.json");
        if (factoryJson.exists()) {
            try {
                factory = (SegmentizerFactory)this.jsonMapper.readValue(factoryJson, SegmentizerFactory.class);
            }
            catch (IOException e) {
                throw new SegmentLoadingException((Throwable)e, "Failed to get segment facotry for %s", new Object[]{e.getMessage()});
            }
        } else {
            factory = new MMappedQueryableSegmentizerFactory(this.indexIO);
        }
        return factory;
    }

    private File getEffectiveInfoDir() {
        File infoDir;
        if (this.config.getInfoDir() != null) {
            infoDir = this.config.getInfoDir();
        } else if (!this.config.getLocations().isEmpty()) {
            infoDir = new File(this.config.getLocations().get(0).getPath(), "info_dir");
        } else if (!this.locations.isEmpty()) {
            infoDir = new File(this.locations.get(0).getPath(), "info_dir");
        } else {
            throw DruidException.forPersona((DruidException.Persona)DruidException.Persona.OPERATOR).ofCategory(DruidException.Category.NOT_FOUND).build("Could not determine infoDir. Make sure 'druid.segmentCache.infoDir' or 'druid.segmentCache.locations' is set correctly.", new Object[0]);
        }
        return infoDir;
    }

    private static String getSegmentDir(DataSegment segment) {
        return DataSegmentPusher.getDefaultStorageDir((DataSegment)segment, (boolean)false);
    }

    boolean isSegmentCached(DataSegment segment) {
        return this.findStoragePathIfCached(segment) != null;
    }

    @Nullable
    private File findStoragePathIfCached(DataSegment segment) {
        for (StorageLocation location : this.locations) {
            String storageDir;
            File localStorageDir = location.segmentDirectoryAsFile(storageDir = SegmentLocalCacheManager.getSegmentDir(segment));
            if (!localStorageDir.exists()) continue;
            if (this.checkSegmentFilesIntact(localStorageDir)) {
                log.warn("[%s] may be damaged. Delete all the segment files and pull from DeepStorage again.", new Object[]{localStorageDir.getAbsolutePath()});
                this.cleanupCacheFiles(location.getPath(), localStorageDir);
                location.removeSegmentDir(localStorageDir, segment);
                break;
            }
            location.maybeReserve(storageDir, segment);
            return localStorageDir;
        }
        return null;
    }

    private boolean checkSegmentFilesIntact(File dir) {
        return this.checkSegmentFilesIntactWithStartMarker(dir);
    }

    private boolean checkSegmentFilesIntactWithStartMarker(File localStorageDir) {
        File downloadStartMarker = new File(localStorageDir.getPath(), DOWNLOAD_START_MARKER_FILE_NAME);
        return downloadStartMarker.exists();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public File getSegmentFiles(DataSegment segment) throws SegmentLoadingException {
        ReferenceCountingLock lock;
        ReferenceCountingLock referenceCountingLock = lock = this.createOrGetLock(segment);
        synchronized (referenceCountingLock) {
            try {
                File segmentDir = this.findStoragePathIfCached(segment);
                if (segmentDir != null) {
                    File file = segmentDir;
                    return file;
                }
                File file = this.loadSegmentWithRetry(segment);
                return file;
            }
            finally {
                this.unlock(segment, lock);
            }
        }
    }

    private File loadSegmentWithRetry(DataSegment segment) throws SegmentLoadingException {
        String segmentDir = SegmentLocalCacheManager.getSegmentDir(segment);
        for (StorageLocation loc : this.locations) {
            if (!loc.isReserved(segmentDir)) continue;
            File storageDir = loc.segmentDirectoryAsFile(segmentDir);
            boolean success = this.loadInLocationWithStartMarkerQuietly(loc, segment, storageDir, false);
            if (!success) {
                throw new SegmentLoadingException("Failed to load segment[%s] in reserved location[%s]", new Object[]{segment.getId(), loc.getPath().getAbsolutePath()});
            }
            return storageDir;
        }
        Iterator<StorageLocation> locationsIterator = this.strategy.getLocations();
        while (locationsIterator.hasNext()) {
            boolean success;
            StorageLocation loc;
            loc = locationsIterator.next();
            File storageDir = loc.reserve(segmentDir, segment);
            if (storageDir == null || !(success = this.loadInLocationWithStartMarkerQuietly(loc, segment, storageDir, true))) continue;
            return storageDir;
        }
        throw new SegmentLoadingException("Failed to load segment[%s] in all locations.", new Object[]{segment.getId()});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean loadInLocationWithStartMarkerQuietly(StorageLocation loc, DataSegment segment, File storageDir, boolean releaseLocation) {
        try {
            this.loadInLocationWithStartMarker(segment, storageDir);
            return true;
        }
        catch (SegmentLoadingException e) {
            try {
                log.makeAlert((Throwable)e, "Failed to load segment in current location [%s], try next location if any", new Object[]{loc.getPath().getAbsolutePath()}).addData("location", (Object)loc.getPath().getAbsolutePath()).emit();
            }
            finally {
                if (releaseLocation) {
                    loc.removeSegmentDir(storageDir, segment);
                }
                this.cleanupCacheFiles(loc.getPath(), storageDir);
            }
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void loadInLocationWithStartMarker(DataSegment segment, File storageDir) throws SegmentLoadingException {
        File downloadStartMarker = new File(storageDir, DOWNLOAD_START_MARKER_FILE_NAME);
        Object object = this.directoryWriteRemoveLock;
        synchronized (object) {
            try {
                FileUtils.mkdirp((File)storageDir);
                if (!downloadStartMarker.createNewFile()) {
                    throw new SegmentLoadingException("Was not able to create new download marker for [%s]", new Object[]{storageDir});
                }
            }
            catch (IOException e) {
                throw new SegmentLoadingException((Throwable)e, "Unable to create marker file for [%s]", new Object[]{storageDir});
            }
        }
        this.loadInLocation(segment, storageDir);
        if (!downloadStartMarker.delete()) {
            throw new SegmentLoadingException("Unable to remove marker file for [%s]", new Object[]{storageDir});
        }
    }

    private void loadInLocation(DataSegment segment, File storageDir) throws SegmentLoadingException {
        LoadSpec loadSpec = (LoadSpec)this.jsonMapper.convertValue((Object)segment.getLoadSpec(), LoadSpec.class);
        LoadSpec.LoadSpecResult result = loadSpec.loadSegment(storageDir);
        if (result.getSize() != segment.getSize()) {
            log.warn("Segment [%s] is different than expected size. Expected [%d] found [%d]", new Object[]{segment.getId(), segment.getSize(), result.getSize()});
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public boolean reserve(DataSegment segment) {
        ReferenceCountingLock lock;
        ReferenceCountingLock referenceCountingLock = lock = this.createOrGetLock(segment);
        synchronized (referenceCountingLock) {
            try {
                StorageLocation location2;
                if (null != this.findStoragePathIfCached(segment)) {
                    boolean bl = true;
                    return bl;
                }
                String storageDirStr = SegmentLocalCacheManager.getSegmentDir(segment);
                for (StorageLocation location2 : this.locations) {
                    if (!location2.isReserved(storageDirStr)) continue;
                    boolean bl = true;
                    return bl;
                }
                Iterator<StorageLocation> it = this.strategy.getLocations();
                do {
                    if (!it.hasNext()) return false;
                } while (null == (location2 = it.next()).reserve(storageDirStr, segment));
                boolean bl = true;
                return bl;
            }
            finally {
                this.unlock(segment, lock);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public boolean release(DataSegment segment) {
        ReferenceCountingLock lock;
        ReferenceCountingLock referenceCountingLock = lock = this.createOrGetLock(segment);
        synchronized (referenceCountingLock) {
            try {
                StorageLocation location;
                String storageDir = SegmentLocalCacheManager.getSegmentDir(segment);
                Iterator<StorageLocation> iterator = this.locations.iterator();
                do {
                    if (!iterator.hasNext()) return false;
                } while (!(location = iterator.next()).isReserved(storageDir));
                File localStorageDir = location.segmentDirectoryAsFile(storageDir);
                if (localStorageDir.exists()) {
                    throw new ISE("Asking to release a location '%s' while the segment directory '%s' is present on disk. Any state on disk must be deleted before releasing", new Object[]{location.getPath().getAbsolutePath(), localStorageDir.getAbsolutePath()});
                }
                boolean bl = location.release(storageDir, segment.getSize());
                return bl;
            }
            finally {
                this.unlock(segment, lock);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void cleanup(DataSegment segment) {
        ReferenceCountingLock lock;
        if (!this.config.isDeleteOnRemove()) {
            return;
        }
        ReferenceCountingLock referenceCountingLock = lock = this.createOrGetLock(segment);
        synchronized (referenceCountingLock) {
            try {
                File loc = this.findStoragePathIfCached(segment);
                if (loc == null) {
                    log.warn("Asked to cleanup something[%s] that didn't exist.  Skipping.", new Object[]{segment.getId()});
                    return;
                }
                for (StorageLocation location : this.locations) {
                    File localStorageDir = new File(location.getPath(), SegmentLocalCacheManager.getSegmentDir(segment));
                    if (!localStorageDir.exists()) continue;
                    this.cleanupCacheFiles(location.getPath(), localStorageDir);
                    location.removeSegmentDir(localStorageDir, segment);
                }
            }
            finally {
                this.unlock(segment, lock);
            }
        }
    }

    @Override
    public void loadSegmentIntoPageCache(DataSegment segment) {
        if (this.loadOnDownloadExec == null) {
            return;
        }
        this.loadOnDownloadExec.submit(() -> this.loadSegmentIntoPageCacheInternal(segment));
    }

    @Override
    public void loadSegmentIntoPageCacheOnBootstrap(DataSegment segment) {
        if (this.loadOnBootstrapExec == null) {
            return;
        }
        this.loadOnBootstrapExec.submit(() -> this.loadSegmentIntoPageCacheInternal(segment));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void loadSegmentIntoPageCacheInternal(DataSegment segment) {
        ReferenceCountingLock lock;
        ReferenceCountingLock referenceCountingLock = lock = this.createOrGetLock(segment);
        synchronized (referenceCountingLock) {
            try {
                for (StorageLocation location : this.locations) {
                    File baseFile;
                    File localStorageDir = new File(location.getPath(), DataSegmentPusher.getDefaultStorageDir((DataSegment)segment, (boolean)false));
                    if (!localStorageDir.exists() || localStorageDir.equals(baseFile = location.getPath())) continue;
                    log.info("Loading directory[%s] into page cache.", new Object[]{localStorageDir});
                    File[] children = localStorageDir.listFiles();
                    if (children == null) continue;
                    for (File child : children) {
                        try (InputStream in = Files.newInputStream(child.toPath(), new OpenOption[0]);){
                            IOUtils.copy((InputStream)in, (OutputStream)NullOutputStream.NULL_OUTPUT_STREAM);
                            log.info("Loaded [%s] into page cache.", new Object[]{child.getAbsolutePath()});
                        }
                        catch (Exception e) {
                            log.error((Throwable)e, "Failed to load [%s] into page cache", new Object[]{child.getAbsolutePath()});
                        }
                    }
                }
            }
            finally {
                this.unlock(segment, lock);
            }
        }
    }

    @Override
    public void shutdownBootstrap() {
        if (this.loadOnBootstrapExec == null) {
            return;
        }
        this.loadOnBootstrapExec.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void cleanupCacheFiles(File baseFile, File cacheFile) {
        if (cacheFile.equals(baseFile)) {
            return;
        }
        Object object = this.directoryWriteRemoveLock;
        synchronized (object) {
            File[] children;
            log.info("Deleting directory[%s]", new Object[]{cacheFile});
            try {
                FileUtils.deleteDirectory((File)cacheFile);
            }
            catch (Exception e) {
                log.error((Throwable)e, "Unable to remove directory[%s]", new Object[]{cacheFile});
            }
            File parent = cacheFile.getParentFile();
            if (parent != null && ((children = parent.listFiles()) == null || children.length == 0)) {
                this.cleanupCacheFiles(baseFile, parent);
            }
        }
    }

    private ReferenceCountingLock createOrGetLock(DataSegment dataSegment) {
        return this.segmentLocks.compute(dataSegment, (segment, lock) -> {
            ReferenceCountingLock nonNullLock = lock == null ? new ReferenceCountingLock() : lock;
            nonNullLock.increment();
            return nonNullLock;
        });
    }

    private void unlock(DataSegment dataSegment, ReferenceCountingLock lock) {
        this.segmentLocks.compute(dataSegment, (segment, existingLock) -> {
            if (existingLock == null) {
                throw new ISE("Lock has already been removed", new Object[0]);
            }
            if (existingLock != lock) {
                throw new ISE("Different lock instance", new Object[0]);
            }
            if (((ReferenceCountingLock)existingLock).numReferences == 1) {
                return null;
            }
            ((ReferenceCountingLock)existingLock).decrement();
            return existingLock;
        });
    }

    @VisibleForTesting
    public ConcurrentHashMap<DataSegment, ReferenceCountingLock> getSegmentLocks() {
        return this.segmentLocks;
    }

    @VisibleForTesting
    public List<StorageLocation> getLocations() {
        return this.locations;
    }

    private static class ReferenceCountingLock {
        private int numReferences;

        private ReferenceCountingLock() {
        }

        private void increment() {
            ++this.numReferences;
        }

        private void decrement() {
            --this.numReferences;
        }
    }
}

