/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSInputChecker;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.BlockReader;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.BlockLocalPathInfo;
import org.apache.hadoop.hdfs.protocol.ClientDatanodeProtocol;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenIdentifier;
import org.apache.hadoop.hdfs.server.datanode.BlockMetadataHeader;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.ipc.RPC;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.util.DataChecksum;

class BlockReaderLocal
extends FSInputChecker
implements BlockReader {
    public static final Log LOG = LogFactory.getLog(DFSClient.class);
    private static Map<Integer, LocalDatanodeInfo> localDatanodeInfoMap = new HashMap<Integer, LocalDatanodeInfo>();
    private FileInputStream dataIn;
    private FileInputStream checksumIn;
    private DataChecksum checksum;
    private int bytesPerChecksum;
    private int checksumSize;
    private long firstChunkOffset;
    private long lastChunkLen = -1L;
    private long lastChunkOffset = -1L;
    private long startOffset;
    private boolean gotEOS = false;
    private byte[] skipBuf = null;

    static BlockReaderLocal newBlockReader(Configuration conf, String file, Block blk, Token<BlockTokenIdentifier> token, DatanodeInfo node, int socketTimeout, long startOffset, long length, boolean connectToDnViaHostname) throws IOException {
        LocalDatanodeInfo localDatanodeInfo = BlockReaderLocal.getLocalDatanodeInfo(node.getIpcPort());
        BlockLocalPathInfo pathinfo = localDatanodeInfo.getBlockLocalPathInfo(blk);
        if (pathinfo == null) {
            pathinfo = BlockReaderLocal.getBlockPathInfo(blk, node, conf, socketTimeout, token, connectToDnViaHostname);
        }
        FileInputStream dataIn = null;
        FileInputStream checksumIn = null;
        BlockReaderLocal localBlockReader = null;
        boolean skipChecksum = BlockReaderLocal.skipChecksumCheck(conf);
        try {
            File blkfile = new File(pathinfo.getBlockPath());
            dataIn = new FileInputStream(blkfile);
            if (LOG.isDebugEnabled()) {
                LOG.debug("New BlockReaderLocal for file " + blkfile + " of size " + blkfile.length() + " startOffset " + startOffset + " length " + length + " short circuit checksum " + skipChecksum);
            }
            if (!skipChecksum) {
                File metafile = new File(pathinfo.getMetaPath());
                checksumIn = new FileInputStream(metafile);
                BlockMetadataHeader header = BlockMetadataHeader.readHeader(new DataInputStream(checksumIn));
                short version = header.getVersion();
                if (version != 1) {
                    LOG.warn("Wrong version (" + version + ") for metadata file for " + blk + " ignoring ...");
                }
                DataChecksum checksum = header.getChecksum();
                localBlockReader = new BlockReaderLocal(conf, file, blk, token, startOffset, length, pathinfo, checksum, true, dataIn, checksumIn);
            } else {
                localBlockReader = new BlockReaderLocal(conf, file, blk, token, startOffset, length, pathinfo, dataIn);
            }
        }
        catch (IOException e) {
            localDatanodeInfo.removeBlockLocalPathInfo(blk);
            DFSClient.LOG.warn("BlockReaderLocal: Removing " + blk + " from cache because local file " + pathinfo.getBlockPath() + " could not be opened.");
            throw e;
        }
        finally {
            if (localBlockReader == null) {
                if (dataIn != null) {
                    dataIn.close();
                }
                if (checksumIn != null) {
                    checksumIn.close();
                }
            }
        }
        return localBlockReader;
    }

    private static synchronized LocalDatanodeInfo getLocalDatanodeInfo(int port) {
        LocalDatanodeInfo ldInfo = localDatanodeInfoMap.get(port);
        if (ldInfo == null) {
            ldInfo = new LocalDatanodeInfo();
            localDatanodeInfoMap.put(port, ldInfo);
        }
        return ldInfo;
    }

    private static BlockLocalPathInfo getBlockPathInfo(Block blk, DatanodeInfo node, Configuration conf, int timeout, Token<BlockTokenIdentifier> token, boolean connectToDnViaHostname) throws IOException {
        LocalDatanodeInfo localDatanodeInfo = BlockReaderLocal.getLocalDatanodeInfo(node.ipcPort);
        BlockLocalPathInfo pathinfo = null;
        ClientDatanodeProtocol proxy = localDatanodeInfo.getDatanodeProxy(node, conf, timeout, connectToDnViaHostname);
        try {
            pathinfo = proxy.getBlockLocalPathInfo(blk, token);
            if (pathinfo != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Cached location of block " + blk + " as " + pathinfo);
                }
                localDatanodeInfo.setBlockLocalPathInfo(blk, pathinfo);
            }
        }
        catch (IOException e) {
            localDatanodeInfo.resetDatanodeProxy();
            throw e;
        }
        return pathinfo;
    }

    private static boolean skipChecksumCheck(Configuration conf) {
        return conf.getBoolean("dfs.client.read.shortcircuit.skip.checksum", false);
    }

    private BlockReaderLocal(Configuration conf, String hdfsfile, Block block, Token<BlockTokenIdentifier> token, long startOffset, long length, BlockLocalPathInfo pathinfo, FileInputStream dataIn) throws IOException {
        super(new Path("/blk_" + block.getBlockId() + ":of:" + hdfsfile), 1);
        long skipped;
        this.startOffset = startOffset;
        this.dataIn = dataIn;
        for (long toSkip = startOffset; toSkip > 0L; toSkip -= skipped) {
            skipped = dataIn.skip(toSkip);
            if (skipped != 0L) continue;
            throw new IOException("Couldn't initialize input stream");
        }
    }

    private BlockReaderLocal(Configuration conf, String hdfsfile, Block block, Token<BlockTokenIdentifier> token, long startOffset, long length, BlockLocalPathInfo pathinfo, DataChecksum checksum, boolean verifyChecksum, FileInputStream dataIn, FileInputStream checksumIn) throws IOException {
        super(new Path("/blk_" + block.getBlockId() + ":of:" + hdfsfile), 1, verifyChecksum, checksum.getChecksumSize() > 0 ? checksum : null, checksum.getBytesPerChecksum(), checksum.getChecksumSize());
        this.startOffset = startOffset;
        this.dataIn = dataIn;
        this.checksumIn = checksumIn;
        this.checksum = checksum;
        long blockLength = pathinfo.getNumBytes();
        this.bytesPerChecksum = checksum.getBytesPerChecksum();
        if (this.bytesPerChecksum > 0xA00000 && (long)this.bytesPerChecksum > blockLength) {
            checksum = DataChecksum.newDataChecksum(checksum.getChecksumType(), Math.max((int)blockLength, 0xA00000));
            this.bytesPerChecksum = checksum.getBytesPerChecksum();
        }
        this.checksumSize = checksum.getChecksumSize();
        if (startOffset < 0L || startOffset > blockLength || length + startOffset > blockLength) {
            String msg = " Offset " + startOffset + " and length " + length + " don't match block " + block + " ( blockLen " + blockLength + " )";
            LOG.warn("BlockReaderLocal requested with incorrect offset: " + msg);
            throw new IOException(msg);
        }
        this.firstChunkOffset = startOffset - startOffset % (long)this.bytesPerChecksum;
        if (this.firstChunkOffset > 0L) {
            dataIn.getChannel().position(this.firstChunkOffset);
            long checksumSkip = this.firstChunkOffset / (long)this.bytesPerChecksum * (long)this.checksumSize;
            if (checksumSkip > 0L) {
                checksumIn.skip(checksumSkip);
            }
        }
        this.lastChunkOffset = this.firstChunkOffset;
        this.lastChunkLen = -1L;
    }

    @Override
    public synchronized int read(byte[] buf, int off, int len) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("read off " + off + " len " + len);
        }
        if (this.checksum == null) {
            return this.dataIn.read(buf, off, len);
        }
        if (this.lastChunkLen < 0L && this.startOffset > this.firstChunkOffset && len > 0) {
            int toSkip = (int)(this.startOffset - this.firstChunkOffset);
            if (this.skipBuf == null) {
                this.skipBuf = new byte[this.bytesPerChecksum];
            }
            if (super.read(this.skipBuf, 0, toSkip) != toSkip) {
                throw new IOException("Could not skip " + toSkip + " bytes");
            }
        }
        return super.read(buf, off, len);
    }

    @Override
    public int readAll(byte[] buf, int offset, int len) throws IOException {
        return BlockReaderLocal.readFully(this, buf, offset, len);
    }

    @Override
    public synchronized long skip(long n) throws IOException {
        long nSkipped;
        int ret;
        if (LOG.isDebugEnabled()) {
            LOG.debug("skip " + n);
        }
        if (this.checksum == null) {
            return this.dataIn.skip(n);
        }
        if (this.skipBuf == null) {
            this.skipBuf = new byte[this.bytesPerChecksum];
        }
        for (nSkipped = 0L; nSkipped < n; nSkipped += (long)ret) {
            int toSkip = (int)Math.min(n - nSkipped, (long)this.skipBuf.length);
            ret = this.read(this.skipBuf, 0, toSkip);
            if (ret > 0) continue;
            return nSkipped;
        }
        return nSkipped;
    }

    @Override
    public boolean seekToNewSource(long targetPos) throws IOException {
        return false;
    }

    @Override
    public synchronized void seek(long n) throws IOException {
        throw new IOException("Seek() is not supported in BlockReaderLocal");
    }

    @Override
    protected long getChunkPosition(long pos) {
        throw new RuntimeException("getChunkPosition() is not supported, since seek is not implemented");
    }

    @Override
    protected synchronized int readChunk(long pos, byte[] buf, int offset, int len, byte[] checksumBuf) throws IOException {
        int nChecksumRead;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Reading chunk from position " + pos + " at offset " + offset + " with length " + len);
        }
        if (this.gotEOS) {
            this.startOffset = -1L;
            return -1;
        }
        if (checksumBuf.length != this.checksumSize) {
            throw new IOException("Cannot read checksum into buffer. The buffer must be exactly '" + this.checksumSize + "' bytes long to hold the checksum bytes.");
        }
        if (pos + this.firstChunkOffset != this.lastChunkOffset) {
            throw new IOException("Mismatch in pos : " + pos + " + " + this.firstChunkOffset + " != " + this.lastChunkOffset);
        }
        int nRead = this.dataIn.read(buf, offset, this.bytesPerChecksum);
        if (nRead < this.bytesPerChecksum) {
            this.gotEOS = true;
        }
        this.lastChunkOffset += (long)nRead;
        this.lastChunkLen = nRead;
        if (this.checksumIn != null && (nChecksumRead = this.checksumIn.read(checksumBuf)) != this.checksumSize) {
            throw new IOException("Could not read checksum at offset " + this.checksumIn.getChannel().position() + " from the meta file.");
        }
        return nRead;
    }

    @Override
    public synchronized void close() throws IOException {
        IOUtils.closeStream(this.dataIn);
        IOUtils.closeStream(this.checksumIn);
    }

    private static class LocalDatanodeInfo {
        private ClientDatanodeProtocol proxy = null;
        private final Map<Block, BlockLocalPathInfo> cache;

        LocalDatanodeInfo() {
            int cacheSize = 10000;
            float hashTableLoadFactor = 0.75f;
            int hashTableCapacity = (int)Math.ceil(13333.3330078125) + 1;
            this.cache = Collections.synchronizedMap(new LinkedHashMap<Block, BlockLocalPathInfo>(hashTableCapacity, 0.75f, true){
                private static final long serialVersionUID = 1L;

                @Override
                protected boolean removeEldestEntry(Map.Entry<Block, BlockLocalPathInfo> eldest) {
                    return this.size() > 10000;
                }
            });
        }

        private synchronized ClientDatanodeProtocol getDatanodeProxy(DatanodeInfo node, Configuration conf, int socketTimeout, boolean connectToDnViaHostname) throws IOException {
            if (this.proxy == null) {
                this.proxy = DFSClient.createClientDatanodeProtocolProxy(node, conf, socketTimeout, connectToDnViaHostname);
            }
            return this.proxy;
        }

        private synchronized void resetDatanodeProxy() {
            if (null != this.proxy) {
                RPC.stopProxy(this.proxy);
                this.proxy = null;
            }
        }

        private BlockLocalPathInfo getBlockLocalPathInfo(Block b) {
            return this.cache.get(b);
        }

        private void setBlockLocalPathInfo(Block b, BlockLocalPathInfo info) {
            this.cache.put(b, info);
        }

        private void removeBlockLocalPathInfo(Block b) {
            this.cache.remove(b);
        }
    }
}

