/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.net;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.apache.hadoop.hdds.scm.net.NodeSchema;
import org.apache.hadoop.hdds.server.YamlUtils;
import org.apache.ozone.shaded.org.apache.commons.collections4.EnumerationUtils;
import org.apache.ozone.shaded.org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.SAXException;

public final class NodeSchemaLoader {
    private static final Logger LOG = LoggerFactory.getLogger(NodeSchemaLoader.class);
    private static final String CONFIGURATION_TAG = "configuration";
    private static final String LAYOUT_VERSION_TAG = "layoutversion";
    private static final String TOPOLOGY_TAG = "topology";
    private static final String TOPOLOGY_PATH = "path";
    private static final String TOPOLOGY_ENFORCE_PREFIX = "enforceprefix";
    private static final String LAYERS_TAG = "layers";
    private static final String LAYER_TAG = "layer";
    private static final String LAYER_ID = "id";
    private static final String LAYER_TYPE = "type";
    private static final String LAYER_COST = "cost";
    private static final String LAYER_PREFIX = "prefix";
    private static final String LAYER_DEFAULT_NAME = "default";
    private static final int LAYOUT_VERSION = 1;
    private static volatile NodeSchemaLoader instance = null;

    private NodeSchemaLoader() {
    }

    public static NodeSchemaLoader getInstance() {
        if (instance == null) {
            instance = new NodeSchemaLoader();
        }
        return instance;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public NodeSchemaLoadResult loadSchemaFromFile(String schemaFilePath) throws FileNotFoundException {
        try {
            File schemaFile = new File(schemaFilePath);
            if (schemaFile.exists()) {
                LOG.info("Load network topology schema file {}", (Object)schemaFile.getAbsolutePath());
                try (InputStream inputStream2 = Files.newInputStream(schemaFile.toPath(), new OpenOption[0]);){
                    NodeSchemaLoadResult nodeSchemaLoadResult = this.loadSchemaFromStream(schemaFilePath, inputStream2);
                    return nodeSchemaLoadResult;
                }
            }
            ClassLoader classloader = Thread.currentThread().getContextClassLoader();
            if (classloader == null) {
                classloader = NodeSchemaLoader.class.getClassLoader();
            }
            if (classloader != null) {
                try (InputStream stream = classloader.getResourceAsStream(schemaFilePath);){
                    if (stream != null) {
                        if (LOG.isInfoEnabled()) {
                            LOG.info("Loading schema from {}", EnumerationUtils.toList(classloader.getResources(schemaFilePath)));
                        }
                        NodeSchemaLoadResult nodeSchemaLoadResult = this.loadSchemaFromStream(schemaFilePath, stream);
                        return nodeSchemaLoadResult;
                    }
                }
            }
            String msg = "Network topology layer schema file " + schemaFilePath + "[" + schemaFile.getAbsolutePath() + "] is not found.";
            LOG.warn(msg);
            throw new FileNotFoundException(msg);
        }
        catch (FileNotFoundException e) {
            throw e;
        }
        catch (IOException | ParserConfigurationException | SAXException e) {
            throw new IllegalArgumentException("Failed to load network topology node schema file: " + schemaFilePath + " , error:" + e.getMessage(), e);
        }
    }

    private NodeSchemaLoadResult loadSchemaFromStream(String schemaFilePath, InputStream stream) throws ParserConfigurationException, SAXException, IOException {
        if (FilenameUtils.getExtension(schemaFilePath).toLowerCase().compareTo("yaml") == 0) {
            return this.loadSchemaFromYaml(stream);
        }
        return this.loadSchema(stream);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private NodeSchemaLoadResult loadSchema(InputStream inputStream2) throws ParserConfigurationException, SAXException, IOException {
        LOG.info("Loading network topology layer schema file");
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        dbf.setFeature("http://javax.xml.XMLConstants/feature/secure-processing", true);
        dbf.setIgnoringComments(true);
        DocumentBuilder builder = dbf.newDocumentBuilder();
        Document doc = builder.parse(inputStream2);
        Element root = doc.getDocumentElement();
        if (!CONFIGURATION_TAG.equals(root.getTagName())) {
            throw new IllegalArgumentException("Bad network topology layer schema configuration file: top-level element not <configuration>");
        }
        if (root.getElementsByTagName(LAYOUT_VERSION_TAG).getLength() != 1) throw new IllegalArgumentException("Bad network topology layer schema configuration file: no or multiple <layoutversion> elements");
        if (this.loadLayoutVersion(root) != 1) throw new IllegalArgumentException("The parse failed because of bad layoutversion value, expected:1");
        if (root.getElementsByTagName(LAYERS_TAG).getLength() != 1) throw new IllegalArgumentException("Bad network topology layer schema configuration file: no or multiple <layers>element");
        Map<String, NodeSchema> schemas = this.loadLayersSection(root);
        if (root.getElementsByTagName(TOPOLOGY_TAG).getLength() != 1) throw new IllegalArgumentException("Bad network topology layer schema configuration file: no or multiple <topology> element");
        return this.loadTopologySection(root, schemas);
    }

    private NodeSchemaLoadResult loadSchemaFromYaml(InputStream schemaFile) {
        NodeSchemaLoadResult finalSchema;
        LOG.info("Loading network topology layer schema file {}", (Object)schemaFile);
        try {
            NodeSchema nodeTree = YamlUtils.loadAs(schemaFile, NodeSchema.class);
            ArrayList<NodeSchema> schemaList = new ArrayList<NodeSchema>();
            if (nodeTree.getType() != NodeSchema.LayerType.ROOT) {
                throw new IllegalArgumentException("First layer is not a ROOT node. schema file.");
            }
            schemaList.add(nodeTree);
            if (nodeTree.getSublayer() != null) {
                nodeTree = nodeTree.getSublayer().get(0);
            }
            while (nodeTree != null) {
                if (nodeTree.getType() == NodeSchema.LayerType.LEAF_NODE && nodeTree.getSublayer() != null) {
                    throw new IllegalArgumentException("Leaf node in the middle of path. schema file.");
                }
                if (nodeTree.getType() == NodeSchema.LayerType.ROOT) {
                    throw new IllegalArgumentException("Multiple root nodes are defined. schema file.");
                }
                schemaList.add(nodeTree);
                if (nodeTree.getSublayer() == null) break;
                nodeTree = nodeTree.getSublayer().get(0);
            }
            finalSchema = new NodeSchemaLoadResult(schemaList, true);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Fail to load network topology node schema file: " + schemaFile + " , error:" + e.getMessage(), e);
        }
        return finalSchema;
    }

    private int loadLayoutVersion(Element root) {
        int layoutVersion;
        Text text = (Text)root.getElementsByTagName(LAYOUT_VERSION_TAG).item(0).getFirstChild();
        if (text != null) {
            String value = text.getData().trim();
            try {
                layoutVersion = Integer.parseInt(value);
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("Bad layoutversion value " + value + " is found. It should be an integer.");
            }
        } else {
            throw new IllegalArgumentException("Value of <layoutversion> is null");
        }
        return layoutVersion;
    }

    private Map<String, NodeSchema> loadLayersSection(Element root) {
        NodeList elements = root.getElementsByTagName(LAYER_TAG);
        HashMap<String, NodeSchema> schemas = new HashMap<String, NodeSchema>();
        for (int i = 0; i < elements.getLength(); ++i) {
            Node node = elements.item(i);
            if (!(node instanceof Element)) continue;
            Element element = (Element)node;
            if (LAYER_TAG.equals(element.getTagName())) {
                String layerId = element.getAttribute(LAYER_ID);
                NodeSchema schema = this.parseLayerElement(element);
                if (!schemas.containsValue(schema)) {
                    schemas.put(layerId, schema);
                    continue;
                }
                throw new IllegalArgumentException("Repetitive layer in network topology node schema configuration file: " + layerId);
            }
            throw new IllegalArgumentException("Bad element in network topology node schema configuration file: " + element.getTagName());
        }
        boolean foundRoot = false;
        boolean foundLeaf = false;
        for (NodeSchema schema : schemas.values()) {
            if (schema.getType() == NodeSchema.LayerType.ROOT) {
                if (foundRoot) {
                    throw new IllegalArgumentException("Multiple ROOT layers are found in network topology schema configuration file");
                }
                foundRoot = true;
            }
            if (schema.getType() != NodeSchema.LayerType.LEAF_NODE) continue;
            if (foundLeaf) {
                throw new IllegalArgumentException("Multiple LEAF layers are found in network topology schema configuration file");
            }
            foundLeaf = true;
        }
        if (!foundRoot) {
            throw new IllegalArgumentException("No ROOT layer is found in network topology schema configuration file");
        }
        if (!foundLeaf) {
            throw new IllegalArgumentException("No LEAF layer is found in network topology schema configuration file");
        }
        return schemas;
    }

    private NodeSchemaLoadResult loadTopologySection(Element root, Map<String, NodeSchema> schemas) {
        NodeList elements = root.getElementsByTagName(TOPOLOGY_TAG).item(0).getChildNodes();
        ArrayList<NodeSchema> schemaList = new ArrayList<NodeSchema>();
        boolean enforcePrefix = false;
        for (int i = 0; i < elements.getLength(); ++i) {
            String value;
            Node node = elements.item(i);
            if (!(node instanceof Element)) continue;
            Element element = (Element)node;
            String tagName = element.getTagName();
            Text text = (Text)element.getFirstChild();
            if (text != null) {
                value = text.getData().trim();
                if (value.isEmpty()) {
                    continue;
                }
            } else {
                throw new IllegalArgumentException("Value of <" + tagName + "> is null");
            }
            if (TOPOLOGY_PATH.equals(tagName)) {
                String[] layerIDs;
                if (value.startsWith("/")) {
                    value = value.substring(1);
                }
                if ((layerIDs = value.split("/")).length != schemas.size()) {
                    throw new IllegalArgumentException("Topology path depth doesn't match layer element numbers");
                }
                for (String layerID : layerIDs) {
                    if (schemas.get(layerID) != null) continue;
                    throw new IllegalArgumentException("No layer found for id " + layerID);
                }
                if (schemas.get(layerIDs[0]).getType() != NodeSchema.LayerType.ROOT) {
                    throw new IllegalArgumentException("Topology path doesn't start with ROOT layer");
                }
                if (schemas.get(layerIDs[layerIDs.length - 1]).getType() != NodeSchema.LayerType.LEAF_NODE) {
                    throw new IllegalArgumentException("Topology path doesn't end with LEAF layer");
                }
                for (String layerID : layerIDs) {
                    schemaList.add(schemas.get(layerID));
                }
                continue;
            }
            if (TOPOLOGY_ENFORCE_PREFIX.equalsIgnoreCase(tagName)) {
                enforcePrefix = Boolean.parseBoolean(value);
                continue;
            }
            throw new IllegalArgumentException("Unsupported Element <" + tagName + ">");
        }
        if (enforcePrefix) {
            for (NodeSchema schema : schemas.values()) {
                if (schema.getType() != NodeSchema.LayerType.INNER_NODE || schema.getPrefix() != null) continue;
                throw new IllegalArgumentException("There is layer without prefix defined while prefix is enforced.");
            }
        }
        return new NodeSchemaLoadResult(schemaList, enforcePrefix);
    }

    private NodeSchema parseLayerElement(Element element) {
        NodeList fields = element.getChildNodes();
        NodeSchema.LayerType type = null;
        int cost = 0;
        String prefix = null;
        String defaultName = null;
        for (int i = 0; i < fields.getLength(); ++i) {
            String value;
            Node fieldNode = fields.item(i);
            if (!(fieldNode instanceof Element)) continue;
            Element field = (Element)fieldNode;
            String tagName = field.getTagName();
            Text text = (Text)field.getFirstChild();
            if (text == null || (value = text.getData().trim()).isEmpty()) continue;
            if (LAYER_COST.equalsIgnoreCase(tagName)) {
                cost = Integer.parseInt(value);
                if (cost >= 0) continue;
                throw new IllegalArgumentException("Cost should be positive number or 0");
            }
            if (LAYER_TYPE.equalsIgnoreCase(tagName)) {
                type = NodeSchema.LayerType.getType(value);
                if (type != null) continue;
                throw new IllegalArgumentException("Unsupported layer type:" + value);
            }
            if (LAYER_PREFIX.equalsIgnoreCase(tagName)) {
                prefix = value;
                continue;
            }
            if (LAYER_DEFAULT_NAME.equalsIgnoreCase(tagName)) {
                defaultName = value;
                continue;
            }
            throw new IllegalArgumentException("Unsupported Element <" + tagName + ">");
        }
        if (type == null) {
            throw new IllegalArgumentException("Missing type Element");
        }
        return new NodeSchema(type, cost, prefix, defaultName);
    }

    public static class NodeSchemaLoadResult {
        private List<NodeSchema> schemaList;
        private boolean enforcePrefix;

        NodeSchemaLoadResult(List<NodeSchema> schemaList, boolean enforcePrefix) {
            this.schemaList = schemaList;
            this.enforcePrefix = enforcePrefix;
        }

        public boolean isEnforePrefix() {
            return this.enforcePrefix;
        }

        public List<NodeSchema> getSchemaList() {
            return this.schemaList;
        }
    }
}

