/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.storage;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Supplier;
import org.apache.hadoop.fs.ByteBufferReadable;
import org.apache.hadoop.fs.CanUnbuffer;
import org.apache.hadoop.fs.Seekable;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.XceiverClientFactory;
import org.apache.hadoop.hdds.scm.XceiverClientSpi;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.storage.ContainerProtocolCalls;
import org.apache.hadoop.ozone.common.Checksum;
import org.apache.hadoop.ozone.common.ChecksumData;
import org.apache.hadoop.ozone.common.OzoneChecksumException;
import org.apache.hadoop.ozone.common.utils.BufferUtils;
import org.apache.hadoop.security.token.Token;
import org.apache.ozone.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.ozone.shaded.com.google.common.base.Preconditions;
import org.apache.ozone.shaded.org.apache.commons.lang3.tuple.Pair;
import org.apache.ozone.shaded.org.apache.ratis.thirdparty.com.google.protobuf.ByteString;

public class ChunkInputStream
extends InputStream
implements Seekable,
CanUnbuffer,
ByteBufferReadable {
    private final ContainerProtos.ChunkInfo chunkInfo;
    private final long length;
    private final BlockID blockID;
    private ContainerProtos.DatanodeBlockID datanodeBlockID;
    private final XceiverClientFactory xceiverClientFactory;
    private XceiverClientSpi xceiverClient;
    private final Supplier<Pipeline> pipelineSupplier;
    private final boolean verifyChecksum;
    private boolean allocated = false;
    private ByteBuffer[] buffers;
    private int bufferIndex;
    private long[] bufferOffsets = null;
    private long bufferOffsetWrtChunkData;
    private int firstUnreleasedBufferIndex = 0;
    private long buffersSize;
    private long chunkPosition = -1L;
    private final Supplier<Token<?>> tokenSupplier;
    private static final int EOF = -1;
    private final List<XceiverClientSpi.Validator> validators;

    ChunkInputStream(ContainerProtos.ChunkInfo chunkInfo, BlockID blockId, XceiverClientFactory xceiverClientFactory, Supplier<Pipeline> pipelineSupplier, boolean verifyChecksum, Supplier<Token<?>> tokenSupplier) {
        this.chunkInfo = chunkInfo;
        this.length = chunkInfo.getLen();
        this.blockID = blockId;
        this.xceiverClientFactory = xceiverClientFactory;
        this.pipelineSupplier = pipelineSupplier;
        this.verifyChecksum = verifyChecksum;
        this.tokenSupplier = tokenSupplier;
        this.validators = ContainerProtocolCalls.toValidatorList(this::validateChunk);
    }

    public synchronized long getRemaining() {
        return this.length - this.getPos();
    }

    @Override
    public synchronized int read() throws IOException {
        this.acquireClient();
        int available = this.prepareRead(1);
        int dataout = -1;
        if (available == -1) {
            Preconditions.checkState(this.buffers == null);
        } else {
            dataout = Byte.toUnsignedInt(this.buffers[this.bufferIndex].get());
        }
        if (this.bufferEOF()) {
            this.releaseBuffers(this.bufferIndex);
        }
        return dataout;
    }

    @Override
    public synchronized int read(byte[] b, int off, int len) throws IOException {
        if (b == null) {
            throw new NullPointerException();
        }
        if (off < 0 || len < 0 || len > b.length - off) {
            throw new IndexOutOfBoundsException();
        }
        if (len == 0) {
            return 0;
        }
        this.acquireClient();
        int total = 0;
        while (len > 0) {
            int available = this.prepareRead(len);
            if (available == -1) {
                Preconditions.checkState(this.buffers == null);
                return total != 0 ? total : -1;
            }
            this.buffers[this.bufferIndex].get(b, off + total, available);
            len -= available;
            total += available;
            if (!this.bufferEOF()) continue;
            this.releaseBuffers(this.bufferIndex);
        }
        return total;
    }

    public synchronized int read(ByteBuffer byteBuffer) throws IOException {
        if (byteBuffer == null) {
            throw new NullPointerException();
        }
        int len = byteBuffer.remaining();
        if (len == 0) {
            return 0;
        }
        this.acquireClient();
        int total = 0;
        while (len > 0) {
            int available = this.prepareRead(len);
            if (available == -1) {
                Preconditions.checkState(this.buffers == null);
                return total != 0 ? total : -1;
            }
            ByteBuffer readBuf = this.buffers[this.bufferIndex];
            ByteBuffer tmpBuf = readBuf.duplicate();
            tmpBuf.limit(tmpBuf.position() + available);
            byteBuffer.put(tmpBuf);
            readBuf.position(tmpBuf.position());
            len -= available;
            total += available;
            if (!this.bufferEOF()) continue;
            this.releaseBuffers(this.bufferIndex);
        }
        return total;
    }

    public synchronized void seek(long pos) throws IOException {
        if (pos < 0L || pos > this.length) {
            if (pos == 0L) {
                return;
            }
            throw new EOFException("EOF encountered at pos: " + pos + " for chunk: " + this.chunkInfo.getChunkName());
        }
        if (this.buffersHavePosition(pos)) {
            this.adjustBufferPosition(pos - this.bufferOffsetWrtChunkData);
        } else {
            this.chunkPosition = pos;
        }
    }

    public synchronized long getPos() {
        if (this.chunkPosition >= 0L) {
            return this.chunkPosition;
        }
        if (this.chunkStreamEOF()) {
            return this.length;
        }
        if (this.buffersHaveData()) {
            return this.bufferOffsetWrtChunkData + this.bufferOffsets[this.bufferIndex] + (long)this.buffers[this.bufferIndex].position();
        }
        if (this.buffersAllocated()) {
            return this.bufferOffsetWrtChunkData + this.buffersSize;
        }
        return 0L;
    }

    public boolean seekToNewSource(long targetPos) {
        return false;
    }

    @Override
    public synchronized void close() {
        this.releaseBuffers();
        this.releaseClient();
    }

    protected synchronized void releaseClient() {
        if (this.xceiverClientFactory != null && this.xceiverClient != null) {
            this.xceiverClientFactory.releaseClientForReadData(this.xceiverClient, false);
            this.xceiverClient = null;
        }
    }

    private void updateDatanodeBlockId(Pipeline pipeline) throws IOException {
        DatanodeDetails closestNode = pipeline.getClosestNode();
        int replicaIdx = pipeline.getReplicaIndex(closestNode);
        ContainerProtos.DatanodeBlockID.Builder builder = this.blockID.getDatanodeBlockIDProtobufBuilder();
        if (replicaIdx > 0) {
            builder.setReplicaIndex(replicaIdx);
        }
        this.datanodeBlockID = builder.build();
    }

    protected synchronized void acquireClient() throws IOException {
        if (this.xceiverClientFactory != null && this.xceiverClient == null) {
            Pipeline pipeline = this.pipelineSupplier.get();
            this.xceiverClient = this.xceiverClientFactory.acquireClientForReadData(pipeline);
            this.updateDatanodeBlockId(pipeline);
        }
    }

    private synchronized int prepareRead(int len) throws IOException {
        while (true) {
            if (this.chunkPosition >= 0L) {
                if (this.buffersHavePosition(this.chunkPosition)) {
                    this.adjustBufferPosition(this.chunkPosition - this.bufferOffsetWrtChunkData);
                } else {
                    this.readChunkFromContainer(len);
                }
            }
            if (this.buffersHaveData()) {
                ByteBuffer bb = this.buffers[this.bufferIndex];
                return Math.min(len, bb.remaining());
            }
            if (!this.dataRemainingInChunk()) break;
            this.readChunkFromContainer(len);
        }
        return -1;
    }

    private synchronized void readChunkFromContainer(int len) throws IOException {
        long adjustedBuffersLen;
        long adjustedBuffersOffset;
        long startByteIndex = this.chunkPosition >= 0L ? this.chunkPosition : this.bufferOffsetWrtChunkData + this.buffersSize;
        this.storePosition();
        if (this.verifyChecksum) {
            Pair<Long, Long> adjustedOffsetAndLength = this.computeChecksumBoundaries(startByteIndex, len);
            adjustedBuffersOffset = adjustedOffsetAndLength.getLeft();
            adjustedBuffersLen = adjustedOffsetAndLength.getRight();
        } else {
            adjustedBuffersOffset = startByteIndex;
            adjustedBuffersLen = len;
        }
        ContainerProtos.ChunkInfo adjustedChunkInfo = ContainerProtos.ChunkInfo.newBuilder(this.chunkInfo).setOffset(this.chunkInfo.getOffset() + adjustedBuffersOffset).setLen(adjustedBuffersLen).build();
        this.readChunkDataIntoBuffers(adjustedChunkInfo);
        this.bufferOffsetWrtChunkData = adjustedBuffersOffset;
        this.adjustBufferPosition(startByteIndex - this.bufferOffsetWrtChunkData);
    }

    private void readChunkDataIntoBuffers(ContainerProtos.ChunkInfo readChunkInfo) throws IOException {
        this.buffers = this.readChunk(readChunkInfo);
        this.buffersSize = readChunkInfo.getLen();
        this.bufferOffsets = new long[this.buffers.length];
        int tempOffset = 0;
        for (int i = 0; i < this.buffers.length; ++i) {
            this.bufferOffsets[i] = tempOffset;
            tempOffset += this.buffers[i].limit();
        }
        this.bufferIndex = 0;
        this.firstUnreleasedBufferIndex = 0;
        this.allocated = true;
    }

    @VisibleForTesting
    protected ByteBuffer[] readChunk(ContainerProtos.ChunkInfo readChunkInfo) throws IOException {
        ContainerProtos.ReadChunkResponseProto readChunkResponse = ContainerProtocolCalls.readChunk(this.xceiverClient, readChunkInfo, this.datanodeBlockID, this.validators, this.tokenSupplier.get());
        if (readChunkResponse.hasData()) {
            return readChunkResponse.getData().asReadOnlyByteBufferList().toArray(new ByteBuffer[0]);
        }
        if (readChunkResponse.hasDataBuffers()) {
            List<ByteString> buffersList = readChunkResponse.getDataBuffers().getBuffersList();
            return BufferUtils.getReadOnlyByteBuffersArray(buffersList);
        }
        throw new IOException("Unexpected error while reading chunk data from container. No data returned.");
    }

    private void validateChunk(ContainerProtos.ContainerCommandRequestProto request, ContainerProtos.ContainerCommandResponseProto response) throws OzoneChecksumException {
        List<ByteString> byteStrings;
        ContainerProtos.ChunkInfo reqChunkInfo = request.getReadChunk().getChunkData();
        ContainerProtos.ReadChunkResponseProto readChunkResponse = response.getReadChunk();
        if (readChunkResponse.hasData()) {
            ByteString byteString = readChunkResponse.getData();
            if ((long)byteString.size() != reqChunkInfo.getLen()) {
                throw new OzoneChecksumException(String.format("Inconsistent read for chunk=%s len=%d bytesRead=%d", reqChunkInfo.getChunkName(), reqChunkInfo.getLen(), byteString.size()));
            }
            byteStrings = new ArrayList<ByteString>();
            byteStrings.add(byteString);
        } else {
            byteStrings = readChunkResponse.getDataBuffers().getBuffersList();
            long buffersLen = BufferUtils.getBuffersLen(byteStrings);
            if (buffersLen != reqChunkInfo.getLen()) {
                throw new OzoneChecksumException(String.format("Inconsistent read for chunk=%s len=%d bytesRead=%d", reqChunkInfo.getChunkName(), reqChunkInfo.getLen(), buffersLen));
            }
        }
        if (this.verifyChecksum) {
            ChecksumData checksumData = ChecksumData.getFromProtoBuf(this.chunkInfo.getChecksumData());
            long relativeOffset = reqChunkInfo.getOffset() - this.chunkInfo.getOffset();
            int bytesPerChecksum = checksumData.getBytesPerChecksum();
            int startIndex = (int)(relativeOffset / (long)bytesPerChecksum);
            Checksum.verifyChecksum(byteStrings, checksumData, startIndex);
        }
    }

    private Pair<Long, Long> computeChecksumBoundaries(long startByteIndex, int dataLen) {
        int bytesPerChecksum = this.chunkInfo.getChecksumData().getBytesPerChecksum();
        long endByteIndex = startByteIndex + (long)dataLen - 1L;
        long adjustedChunkOffset = startByteIndex / (long)bytesPerChecksum * (long)bytesPerChecksum;
        long endIndex = (endByteIndex / (long)bytesPerChecksum + 1L) * (long)bytesPerChecksum;
        long adjustedChunkLen = Math.min(endIndex, this.length) - adjustedChunkOffset;
        return Pair.of(adjustedChunkOffset, adjustedChunkLen);
    }

    private void adjustBufferPosition(long bufferPosition) {
        if (this.bufferIndex >= this.buffers.length) {
            this.bufferIndex = Arrays.binarySearch(this.bufferOffsets, bufferPosition);
        } else if (bufferPosition < this.bufferOffsets[this.bufferIndex]) {
            this.bufferIndex = Arrays.binarySearch(this.bufferOffsets, 0, this.bufferIndex, bufferPosition);
        } else if (bufferPosition >= this.bufferOffsets[this.bufferIndex] + (long)this.buffers[this.bufferIndex].capacity()) {
            this.bufferIndex = Arrays.binarySearch(this.bufferOffsets, this.bufferIndex + 1, this.buffers.length, bufferPosition);
        }
        if (this.bufferIndex < 0) {
            this.bufferIndex = -this.bufferIndex - 2;
        }
        this.buffers[this.bufferIndex].position((int)(bufferPosition - this.bufferOffsets[this.bufferIndex]));
        for (int i = this.bufferIndex + 1; i < this.buffers.length; ++i) {
            this.buffers[i].position(0);
        }
        this.resetPosition();
    }

    @VisibleForTesting
    protected boolean buffersAllocated() {
        return this.buffers != null && this.buffers.length > 0;
    }

    private boolean buffersHaveData() {
        boolean hasData = false;
        if (this.buffersAllocated()) {
            while (this.bufferIndex < this.buffers.length) {
                if (this.buffers[this.bufferIndex] != null && this.buffers[this.bufferIndex].hasRemaining()) {
                    hasData = true;
                    break;
                }
                if (!this.buffersRemaining()) break;
                ++this.bufferIndex;
                Preconditions.checkState(this.bufferIndex < this.buffers.length);
            }
        }
        return hasData;
    }

    private boolean buffersRemaining() {
        return this.bufferIndex < this.buffers.length - 1;
    }

    private boolean buffersHavePosition(long pos) {
        if (this.buffersAllocated()) {
            return pos >= this.bufferOffsetWrtChunkData + this.bufferOffsets[this.firstUnreleasedBufferIndex] && pos < this.bufferOffsetWrtChunkData + this.buffersSize;
        }
        return false;
    }

    private boolean dataRemainingInChunk() {
        long bufferPos = this.chunkPosition >= 0L ? this.chunkPosition : this.bufferOffsetWrtChunkData + this.buffersSize;
        return bufferPos < this.length;
    }

    private boolean bufferEOF() {
        return this.allocated && !this.buffers[this.bufferIndex].hasRemaining();
    }

    private boolean chunkStreamEOF() {
        if (!this.allocated) {
            return false;
        }
        if (this.buffersHaveData() || this.dataRemainingInChunk()) {
            return false;
        }
        Preconditions.checkState(this.bufferOffsetWrtChunkData + this.buffersSize == this.length, "EOF detected but not at the last byte of the chunk");
        return true;
    }

    private void releaseBuffers(int releaseUptoBufferIndex) {
        if (releaseUptoBufferIndex == this.buffers.length - 1) {
            this.chunkPosition = this.bufferOffsetWrtChunkData + this.bufferOffsets[releaseUptoBufferIndex] + (long)this.buffers[releaseUptoBufferIndex].capacity();
            this.releaseBuffers();
        } else {
            for (int i = 0; i <= releaseUptoBufferIndex; ++i) {
                this.buffers[i] = null;
            }
            this.firstUnreleasedBufferIndex = releaseUptoBufferIndex + 1;
        }
    }

    private void releaseBuffers() {
        this.buffers = null;
        this.bufferIndex = 0;
        this.firstUnreleasedBufferIndex = 0;
    }

    void resetPosition() {
        this.chunkPosition = -1L;
    }

    private void storePosition() {
        this.chunkPosition = this.getPos();
    }

    String getChunkName() {
        return this.chunkInfo.getChunkName();
    }

    protected long getLength() {
        return this.length;
    }

    @VisibleForTesting
    protected long getChunkPosition() {
        return this.chunkPosition;
    }

    public synchronized void unbuffer() {
        this.storePosition();
        this.releaseBuffers();
        this.releaseClient();
    }

    @VisibleForTesting
    public ByteBuffer[] getCachedBuffers() {
        return BufferUtils.getReadOnlyByteBuffers(this.buffers);
    }

    public ContainerProtos.ChunkInfo getChunkInfo() {
        return this.chunkInfo;
    }
}

