/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.recon.tasks;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import org.apache.commons.lang3.tuple.Triple;
import org.apache.hadoop.hdds.utils.db.BatchOperation;
import org.apache.hadoop.hdds.utils.db.DBStore;
import org.apache.hadoop.hdds.utils.db.RDBBatchOperation;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.hdds.utils.db.TableIterator;
import org.apache.hadoop.ozone.om.OMMetadataManager;
import org.apache.hadoop.ozone.recon.recovery.ReconOMMetadataManager;
import org.apache.hadoop.ozone.recon.spi.ReconGlobalStatsManager;
import org.apache.hadoop.ozone.recon.tasks.DeletedKeysInsightHandler;
import org.apache.hadoop.ozone.recon.tasks.GlobalStatsValue;
import org.apache.hadoop.ozone.recon.tasks.OMDBUpdateEvent;
import org.apache.hadoop.ozone.recon.tasks.OMUpdateEventBatch;
import org.apache.hadoop.ozone.recon.tasks.OmTableHandler;
import org.apache.hadoop.ozone.recon.tasks.OpenKeysInsightHandler;
import org.apache.hadoop.ozone.recon.tasks.ReconOmTask;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OmTableInsightTask
implements ReconOmTask {
    private static final Logger LOG = LoggerFactory.getLogger(OmTableInsightTask.class);
    private ReconGlobalStatsManager reconGlobalStatsManager;
    private ReconOMMetadataManager reconOMMetadataManager;
    private Map<String, OmTableHandler> tableHandlers;
    private Collection<String> tables;
    private Map<String, Long> objectCountMap;
    private Map<String, Long> unReplicatedSizeMap;
    private Map<String, Long> replicatedSizeMap;

    @Inject
    public OmTableInsightTask(ReconGlobalStatsManager reconGlobalStatsManager, ReconOMMetadataManager reconOMMetadataManager) throws IOException {
        this.reconGlobalStatsManager = reconGlobalStatsManager;
        this.reconOMMetadataManager = reconOMMetadataManager;
        this.tableHandlers = new HashMap<String, OmTableHandler>();
        this.tableHandlers.put("openKeyTable", new OpenKeysInsightHandler());
        this.tableHandlers.put("openFileTable", new OpenKeysInsightHandler());
        this.tableHandlers.put("deletedTable", new DeletedKeysInsightHandler());
    }

    @Override
    public ReconOmTask getStagedTask(ReconOMMetadataManager stagedOmMetadataManager, DBStore stagedReconDbStore) throws IOException {
        ReconGlobalStatsManager stagedGlobalStatsManager = this.reconGlobalStatsManager.getStagedReconGlobalStatsManager(stagedReconDbStore);
        return new OmTableInsightTask(stagedGlobalStatsManager, stagedOmMetadataManager);
    }

    @Override
    public void init() {
        ReconOmTask.super.init();
        this.tables = this.getTaskTables();
        this.objectCountMap = this.initializeCountMap();
        this.unReplicatedSizeMap = this.initializeSizeMap(false);
        this.replicatedSizeMap = this.initializeSizeMap(true);
    }

    @Override
    public ReconOmTask.TaskResult reprocess(OMMetadataManager omMetadataManager) {
        this.init();
        for (String tableName : this.tables) {
            Table table = omMetadataManager.getTable(tableName);
            try {
                Table.KeyValueIterator iterator = table.iterator();
                try {
                    if (this.tableHandlers.containsKey(tableName)) {
                        Triple<Long, Long, Long> details = this.tableHandlers.get(tableName).getTableSizeAndCount((TableIterator<String, ? extends Table.KeyValue<String, ?>>)iterator);
                        this.objectCountMap.put(OmTableInsightTask.getTableCountKeyFromTable(tableName), (Long)details.getLeft());
                        this.unReplicatedSizeMap.put(OmTableInsightTask.getUnReplicatedSizeKeyFromTable(tableName), (Long)details.getMiddle());
                        this.replicatedSizeMap.put(OmTableInsightTask.getReplicatedSizeKeyFromTable(tableName), (Long)details.getRight());
                        continue;
                    }
                    long count = Iterators.size((Iterator)iterator);
                    this.objectCountMap.put(OmTableInsightTask.getTableCountKeyFromTable(tableName), count);
                }
                finally {
                    if (iterator == null) continue;
                    iterator.close();
                }
            }
            catch (IOException ioEx) {
                LOG.error("Unable to populate Table Count in Recon DB.", (Throwable)ioEx);
                return this.buildTaskResult(false);
            }
        }
        if (!this.objectCountMap.isEmpty()) {
            this.writeDataToDB(this.objectCountMap);
        }
        if (!this.unReplicatedSizeMap.isEmpty()) {
            this.writeDataToDB(this.unReplicatedSizeMap);
        }
        if (!this.replicatedSizeMap.isEmpty()) {
            this.writeDataToDB(this.replicatedSizeMap);
        }
        LOG.debug("Completed a 'reprocess' run of OmTableInsightTask.");
        return this.buildTaskResult(true);
    }

    @Override
    public String getTaskName() {
        return "OmTableInsightTask";
    }

    public Collection<String> getTaskTables() {
        return new ArrayList<String>(this.reconOMMetadataManager.listTableNames());
    }

    @Override
    public ReconOmTask.TaskResult process(OMUpdateEventBatch events, Map<String, Integer> subTaskSeekPosMap) {
        if (this.tables == null || this.tables.isEmpty()) {
            this.init();
        }
        Iterator<OMDBUpdateEvent> eventIterator = events.getIterator();
        long startTime = Time.monotonicNow();
        block7: while (eventIterator.hasNext()) {
            OMDBUpdateEvent omdbUpdateEvent = eventIterator.next();
            String tableName = omdbUpdateEvent.getTable();
            if (!this.tables.contains(tableName)) continue;
            try {
                switch (omdbUpdateEvent.getAction()) {
                    case PUT: {
                        this.handlePutEvent(omdbUpdateEvent, tableName);
                        continue block7;
                    }
                    case DELETE: {
                        this.handleDeleteEvent(omdbUpdateEvent, tableName);
                        continue block7;
                    }
                    case UPDATE: {
                        this.handleUpdateEvent(omdbUpdateEvent, tableName);
                        continue block7;
                    }
                }
                LOG.trace("Skipping DB update event : Table: {}, Action: {}", (Object)tableName, (Object)omdbUpdateEvent.getAction());
            }
            catch (Exception e) {
                LOG.error("Unexpected exception while processing the table {}, Action: {}", new Object[]{tableName, omdbUpdateEvent.getAction(), e});
                return this.buildTaskResult(false);
            }
        }
        if (!this.objectCountMap.isEmpty()) {
            this.writeDataToDB(this.objectCountMap);
        }
        if (!this.unReplicatedSizeMap.isEmpty()) {
            this.writeDataToDB(this.unReplicatedSizeMap);
        }
        if (!this.replicatedSizeMap.isEmpty()) {
            this.writeDataToDB(this.replicatedSizeMap);
        }
        LOG.debug("{} successfully processed in {} milliseconds", (Object)this.getTaskName(), (Object)(Time.monotonicNow() - startTime));
        return this.buildTaskResult(true);
    }

    private void handlePutEvent(OMDBUpdateEvent<String, Object> event, String tableName) {
        OmTableHandler tableHandler = this.tableHandlers.get(tableName);
        if (event.getValue() != null) {
            if (tableHandler != null) {
                tableHandler.handlePutEvent(event, tableName, this.objectCountMap, this.unReplicatedSizeMap, this.replicatedSizeMap);
            } else {
                String countKey = OmTableInsightTask.getTableCountKeyFromTable(tableName);
                this.objectCountMap.computeIfPresent(countKey, (k, count) -> count + 1L);
            }
        }
    }

    private void handleDeleteEvent(OMDBUpdateEvent<String, Object> event, String tableName) {
        OmTableHandler tableHandler = this.tableHandlers.get(tableName);
        if (event.getValue() != null) {
            if (tableHandler != null) {
                tableHandler.handleDeleteEvent(event, tableName, this.objectCountMap, this.unReplicatedSizeMap, this.replicatedSizeMap);
            } else {
                this.objectCountMap.computeIfPresent(OmTableInsightTask.getTableCountKeyFromTable(tableName), (k, count) -> count > 0L ? count - 1L : 0L);
            }
        }
    }

    private void handleUpdateEvent(OMDBUpdateEvent<String, Object> event, String tableName) {
        OmTableHandler tableHandler = this.tableHandlers.get(tableName);
        if (event.getValue() != null && tableHandler != null) {
            tableHandler.handleUpdateEvent(event, tableName, this.objectCountMap, this.unReplicatedSizeMap, this.replicatedSizeMap);
        }
    }

    private void writeDataToDB(Map<String, Long> dataMap) {
        try (RDBBatchOperation rdbBatchOperation = new RDBBatchOperation();){
            for (Map.Entry<String, Long> entry : dataMap.entrySet()) {
                String key = entry.getKey();
                Long value = entry.getValue();
                GlobalStatsValue globalStatsValue = new GlobalStatsValue(value);
                this.reconGlobalStatsManager.batchStoreGlobalStats((BatchOperation)rdbBatchOperation, key, globalStatsValue);
            }
            this.reconGlobalStatsManager.commitBatchOperation(rdbBatchOperation);
        }
        catch (IOException e) {
            LOG.error("Failed to write data to RocksDB GlobalStats table", (Throwable)e);
        }
    }

    public HashMap<String, Long> initializeCountMap() {
        HashMap<String, Long> objCountMap = new HashMap<String, Long>(this.tables.size());
        for (String tableName : this.tables) {
            String key = OmTableInsightTask.getTableCountKeyFromTable(tableName);
            objCountMap.put(key, this.getValueForKey(key));
        }
        return objCountMap;
    }

    public HashMap<String, Long> initializeSizeMap(boolean replicated) {
        HashMap<String, Long> sizeCountMap = new HashMap<String, Long>();
        for (Map.Entry<String, OmTableHandler> entry : this.tableHandlers.entrySet()) {
            String tableName = entry.getKey();
            OmTableHandler tableHandler = entry.getValue();
            String key = replicated ? tableHandler.getReplicatedSizeKeyFromTable(tableName) : tableHandler.getUnReplicatedSizeKeyFromTable(tableName);
            sizeCountMap.put(key, this.getValueForKey(key));
        }
        return sizeCountMap;
    }

    public static String getTableCountKeyFromTable(String tableName) {
        return tableName + "Count";
    }

    public static String getReplicatedSizeKeyFromTable(String tableName) {
        return tableName + "ReplicatedDataSize";
    }

    public static String getUnReplicatedSizeKeyFromTable(String tableName) {
        return tableName + "UnReplicatedDataSize";
    }

    private long getValueForKey(String key) {
        try {
            GlobalStatsValue globalStatsValue = this.reconGlobalStatsManager.getGlobalStatsValue(key);
            return globalStatsValue == null ? 0L : globalStatsValue.getValue();
        }
        catch (IOException e) {
            LOG.error("Failed to get value for key {} from RocksDB GlobalStats table", (Object)key, (Object)e);
            return 0L;
        }
    }

    @VisibleForTesting
    public void setTables(Collection<String> tables) {
        this.tables = tables;
    }

    @VisibleForTesting
    public void setObjectCountMap(HashMap<String, Long> objectCountMap) {
        this.objectCountMap = objectCountMap;
    }

    @VisibleForTesting
    public void setUnReplicatedSizeMap(HashMap<String, Long> unReplicatedSizeMap) {
        this.unReplicatedSizeMap = unReplicatedSizeMap;
    }

    @VisibleForTesting
    public void setReplicatedSizeMap(HashMap<String, Long> replicatedSizeMap) {
        this.replicatedSizeMap = replicatedSizeMap;
    }
}

