/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.coordinator.group;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import org.apache.kafka.clients.ClientResponse;
import org.apache.kafka.clients.CommonClientConfigs;
import org.apache.kafka.clients.KafkaClient;
import org.apache.kafka.common.IsolationLevel;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.message.ListOffsetsRequestData;
import org.apache.kafka.common.message.ListOffsetsResponseData;
import org.apache.kafka.common.network.ListenerName;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.AbstractRequest;
import org.apache.kafka.common.requests.ListOffsetsRequest;
import org.apache.kafka.common.requests.ListOffsetsResponse;
import org.apache.kafka.common.utils.ExponentialBackoffManager;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.coordinator.group.PartitionMetadataClient;
import org.apache.kafka.metadata.MetadataCache;
import org.apache.kafka.server.util.InterBrokerSendThread;
import org.apache.kafka.server.util.RequestAndCompletionHandler;
import org.apache.kafka.server.util.timer.Timer;
import org.apache.kafka.server.util.timer.TimerTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NetworkPartitionMetadataClient
implements PartitionMetadataClient {
    private static final Logger log = LoggerFactory.getLogger(NetworkPartitionMetadataClient.class);
    private static final long REQUEST_BACKOFF_MS = 1000L;
    private static final long REQUEST_BACKOFF_MAX_MS = 30000L;
    private static final int MAX_RETRY_ATTEMPTS = 5;
    private final MetadataCache metadataCache;
    private final Supplier<KafkaClient> networkClientSupplier;
    private final Time time;
    private final ListenerName listenerName;
    private final AtomicBoolean initialized = new AtomicBoolean(false);
    private volatile SendThread sendThread;
    private final Timer timer;

    public NetworkPartitionMetadataClient(MetadataCache metadataCache, Supplier<KafkaClient> networkClientSupplier, Time time, ListenerName listenerName, Timer timer) {
        if (metadataCache == null) {
            throw new IllegalArgumentException("MetadataCache must not be null.");
        }
        if (networkClientSupplier == null) {
            throw new IllegalArgumentException("NetworkClientSupplier must not be null.");
        }
        if (time == null) {
            throw new IllegalArgumentException("Time must not be null.");
        }
        if (listenerName == null) {
            throw new IllegalArgumentException("ListenerName must not be null.");
        }
        if (timer == null) {
            throw new IllegalArgumentException("Timer must not be null.");
        }
        this.metadataCache = metadataCache;
        this.networkClientSupplier = networkClientSupplier;
        this.time = time;
        this.listenerName = listenerName;
        this.timer = timer;
    }

    @Override
    public Map<TopicPartition, CompletableFuture<PartitionMetadataClient.OffsetResponse>> listLatestOffsets(Set<TopicPartition> topicPartitions) {
        if (topicPartitions == null || topicPartitions.isEmpty()) {
            return Map.of();
        }
        this.ensureSendThreadInitialized();
        HashMap<TopicPartition, CompletableFuture<PartitionMetadataClient.OffsetResponse>> futures = new HashMap<TopicPartition, CompletableFuture<PartitionMetadataClient.OffsetResponse>>();
        HashMap<Node, List> partitionsByNode = new HashMap<Node, List>();
        for (TopicPartition tp : topicPartitions) {
            Optional leaderNodeOpt = this.metadataCache.getPartitionLeaderEndpoint(tp.topic(), tp.partition(), this.listenerName);
            if (leaderNodeOpt.isEmpty() || ((Node)leaderNodeOpt.get()).isEmpty()) {
                futures.put(tp, CompletableFuture.completedFuture(new PartitionMetadataClient.OffsetResponse(-1L, Errors.LEADER_NOT_AVAILABLE)));
                continue;
            }
            partitionsByNode.computeIfAbsent((Node)leaderNodeOpt.get(), k -> new ArrayList()).add(tp);
        }
        partitionsByNode.forEach((node, partitionsByLeader) -> {
            HashMap<TopicPartition, CompletableFuture<PartitionMetadataClient.OffsetResponse>> partitionFuturesByLeader = new HashMap<TopicPartition, CompletableFuture<PartitionMetadataClient.OffsetResponse>>();
            for (TopicPartition tp : partitionsByLeader) {
                CompletableFuture future = new CompletableFuture();
                futures.put(tp, future);
                partitionFuturesByLeader.put(tp, future);
            }
            ListOffsetsRequest.Builder requestBuilder = this.createListOffsetsRequest((List<TopicPartition>)partitionsByLeader);
            PendingRequest pendingRequest = new PendingRequest((Node)node, (Map<TopicPartition, CompletableFuture<PartitionMetadataClient.OffsetResponse>>)partitionFuturesByLeader, requestBuilder);
            this.sendThread.enqueue(pendingRequest);
        });
        return futures;
    }

    @Override
    public void close() {
        Utils.closeQuietly((AutoCloseable)this.timer, (String)"NetworkPartitionMetadataClient timer");
        if (!this.initialized.get()) {
            return;
        }
        if (this.sendThread != null) {
            try {
                this.sendThread.shutdown();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                log.error("Interrupted while shutting down NetworkPartitionMetadataClient", (Throwable)e);
            }
        }
    }

    void ensureSendThreadInitialized() {
        if (this.initialized.compareAndSet(false, true)) {
            KafkaClient networkClient = this.networkClientSupplier.get();
            this.sendThread = new SendThread("NetworkPartitionMetadataClientSendThread", networkClient, Math.toIntExact(CommonClientConfigs.DEFAULT_SOCKET_CONNECTION_SETUP_TIMEOUT_MAX_MS), this.time);
            this.sendThread.start();
            log.info("NetworkPartitionMetadataClient sendThread initialized and started");
        }
    }

    private ListOffsetsRequest.Builder createListOffsetsRequest(List<TopicPartition> partitions) {
        HashMap topicsMap = new HashMap();
        partitions.forEach(tp -> {
            ListOffsetsRequestData.ListOffsetsTopic topic;
            if (!topicsMap.containsKey(tp.topic())) {
                topic = new ListOffsetsRequestData.ListOffsetsTopic().setName(tp.topic());
                topicsMap.put(tp.topic(), topic);
            }
            topic = (ListOffsetsRequestData.ListOffsetsTopic)topicsMap.get(tp.topic());
            topic.partitions().add(new ListOffsetsRequestData.ListOffsetsPartition().setPartitionIndex(tp.partition()).setTimestamp(-1L).setCurrentLeaderEpoch(-1));
        });
        return ListOffsetsRequest.Builder.forConsumer((boolean)true, (IsolationLevel)IsolationLevel.READ_UNCOMMITTED).setTargetTimes(List.copyOf(topicsMap.values()));
    }

    void handleResponse(PendingRequest pendingRequest, ClientResponse clientResponse) {
        if (this.maybeHandleErrorResponse(pendingRequest, clientResponse)) {
            return;
        }
        log.debug("ListOffsets response received successfully - {}", (Object)clientResponse);
        pendingRequest.backoffManager().resetAttempts();
        ListOffsetsResponse response = (ListOffsetsResponse)clientResponse.responseBody();
        Map<TopicPartition, CompletableFuture<PartitionMetadataClient.OffsetResponse>> partitionFutures = pendingRequest.futures();
        for (ListOffsetsResponseData.ListOffsetsTopicResponse topicResponse : response.topics()) {
            String topicName = topicResponse.name();
            for (ListOffsetsResponseData.ListOffsetsPartitionResponse partitionResponse : topicResponse.partitions()) {
                TopicPartition tp2 = new TopicPartition(topicName, partitionResponse.partitionIndex());
                CompletableFuture<PartitionMetadataClient.OffsetResponse> future2 = partitionFutures.get(tp2);
                if (future2 == null) continue;
                future2.complete(new PartitionMetadataClient.OffsetResponse(partitionResponse.offset(), Errors.forCode((short)partitionResponse.errorCode())));
            }
        }
        partitionFutures.forEach((tp, future) -> {
            if (!future.isDone()) {
                future.complete(new PartitionMetadataClient.OffsetResponse(-1L, Errors.UNKNOWN_TOPIC_OR_PARTITION));
            }
        });
    }

    private boolean maybeHandleErrorResponse(PendingRequest pendingRequest, ClientResponse clientResponse) {
        Errors error;
        Map<TopicPartition, CompletableFuture<PartitionMetadataClient.OffsetResponse>> partitionFutures = pendingRequest.futures();
        boolean shouldRetry = false;
        if (clientResponse == null) {
            log.error("Response for ListOffsets for topicPartitions: {} is null", partitionFutures.keySet());
            error = Errors.UNKNOWN_SERVER_ERROR;
        } else if (clientResponse.authenticationException() != null) {
            log.error("Authentication exception", (Throwable)clientResponse.authenticationException());
            error = Errors.UNKNOWN_SERVER_ERROR;
        } else if (clientResponse.versionMismatch() != null) {
            log.error("Version mismatch exception", (Throwable)clientResponse.versionMismatch());
            error = Errors.UNKNOWN_SERVER_ERROR;
        } else if (clientResponse.wasDisconnected()) {
            log.debug("Response for ListOffsets for TopicPartitions: {} was disconnected - {}.", partitionFutures.keySet(), (Object)clientResponse);
            error = Errors.NETWORK_EXCEPTION;
            shouldRetry = true;
        } else if (clientResponse.wasTimedOut()) {
            log.debug("Response for ListOffsets for TopicPartitions: {} timed out - {}.", partitionFutures.keySet(), (Object)clientResponse);
            error = Errors.REQUEST_TIMED_OUT;
            shouldRetry = true;
        } else if (!clientResponse.hasResponse()) {
            log.error("Response for ListOffsets for TopicPartitions: {} has no response - {}.", partitionFutures.keySet(), (Object)clientResponse);
            error = Errors.UNKNOWN_SERVER_ERROR;
        } else {
            return false;
        }
        if (shouldRetry) {
            ExponentialBackoffManager backoffManager = pendingRequest.backoffManager();
            if (backoffManager.canAttempt()) {
                backoffManager.incrementAttempt();
                long backoffMs = backoffManager.backOff();
                log.debug("Retrying ListOffsets request for TopicPartitions: {} after {} ms (attempt {}/{})", new Object[]{partitionFutures.keySet(), backoffMs, backoffManager.attempts(), 5});
                this.timer.add((TimerTask)new RetryTimerTask(backoffMs, pendingRequest));
                return true;
            }
            log.error("Exhausted max retries ({}) for ListOffsets request for TopicPartitions: {}", (Object)5, partitionFutures.keySet());
        }
        partitionFutures.forEach((tp, future) -> future.complete(new PartitionMetadataClient.OffsetResponse(-1L, error)));
        return true;
    }

    private class SendThread
    extends InterBrokerSendThread {
        private final ConcurrentLinkedQueue<PendingRequest> pendingRequests;

        protected SendThread(String name, KafkaClient networkClient, int requestTimeoutMs, Time time) {
            super(name, networkClient, requestTimeoutMs, time);
            this.pendingRequests = new ConcurrentLinkedQueue();
        }

        public void enqueue(PendingRequest pendingRequest) {
            this.pendingRequests.add(pendingRequest);
            this.wakeup();
        }

        public Collection<RequestAndCompletionHandler> generateRequests() {
            PendingRequest pending;
            ArrayList<RequestAndCompletionHandler> requests = new ArrayList<RequestAndCompletionHandler>();
            while ((pending = this.pendingRequests.poll()) != null) {
                PendingRequest current = pending;
                ListOffsetsRequest.Builder requestBuilder = current.requestBuilder;
                RequestAndCompletionHandler requestHandler = new RequestAndCompletionHandler(NetworkPartitionMetadataClient.this.time.hiResClockMs(), current.node, (AbstractRequest.Builder)requestBuilder, response -> NetworkPartitionMetadataClient.this.handleResponse(current, response));
                requests.add(requestHandler);
            }
            return requests;
        }
    }

    record PendingRequest(Node node, Map<TopicPartition, CompletableFuture<PartitionMetadataClient.OffsetResponse>> futures, ListOffsetsRequest.Builder requestBuilder, ExponentialBackoffManager backoffManager) {
        PendingRequest(Node node, Map<TopicPartition, CompletableFuture<PartitionMetadataClient.OffsetResponse>> futures, ListOffsetsRequest.Builder requestBuilder) {
            this(node, futures, requestBuilder, new ExponentialBackoffManager(5, 1000L, 2, 30000L, 0.2));
        }
    }

    private final class RetryTimerTask
    extends TimerTask {
        private final PendingRequest pendingRequest;

        RetryTimerTask(long delayMs, PendingRequest pendingRequest) {
            super(delayMs);
            this.pendingRequest = pendingRequest;
        }

        public void run() {
            NetworkPartitionMetadataClient.this.sendThread.enqueue(this.pendingRequest);
            NetworkPartitionMetadataClient.this.sendThread.wakeup();
        }
    }
}

