/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.www;

import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Serial;
import java.nio.charset.StandardCharsets;
import org.apache.hop.core.annotations.HopServerServlet;
import org.apache.hop.core.util.Utils;
import org.apache.hop.i18n.BaseMessages;
import org.apache.hop.workflow.WorkflowMeta;
import org.apache.hop.workflow.WorkflowSvgPainter;
import org.apache.hop.workflow.engine.IWorkflowEngine;

@HopServerServlet(id = "workflowImage", name = "Generate a PNG image of a workflow")
public class GetWorkflowImageServlet extends BaseHttpServlet implements IHopServerPlugin {
  @Serial private static final long serialVersionUID = -4365372274638005929L;
  public static final float ZOOM_FACTOR = 1.5f;

  private static final Class<?> PKG = GetPipelineStatusServlet.class;

  public static final String CONTEXT_PATH = "/hop/workflowImage";

  @Override
  public void doGet(HttpServletRequest request, HttpServletResponse response)
      throws ServletException, IOException {
    if (isJettyMode() && !request.getContextPath().startsWith(CONTEXT_PATH)) {
      return;
    }
    if (!supportGraphicEnvironment) {
      response.setStatus(HttpServletResponse.SC_NO_CONTENT);
      return;
    }

    if (log.isDebug()) {
      logDebug(BaseMessages.getString(PKG, "GetWorkflowImageServlet.Log.WorkflowImageRequested"));
    }

    String workflowName = request.getParameter("name");
    String id = request.getParameter("id");

    // ID is optional...
    //
    IWorkflowEngine<WorkflowMeta> workflow;
    HopServerObjectEntry entry;
    if (Utils.isEmpty(id)) {
      // get the first pipeline that matches...
      //
      entry = getWorkflowMap().getFirstHopServerObjectEntry(workflowName);
      if (entry == null) {
        workflow = null;
      } else {
        id = entry.getId();
        workflow = getWorkflowMap().getWorkflow(entry);
      }
    } else {
      // Take the ID into account!
      //
      entry = new HopServerObjectEntry(workflowName, id);
      workflow = getWorkflowMap().getWorkflow(entry);
    }

    ByteArrayOutputStream svgStream = null;

    try {
      if (workflow != null) {

        response.setStatus(HttpServletResponse.SC_OK);

        response.setCharacterEncoding("UTF-8");
        response.setContentType("image/svg+xml");

        // Generate workflow SVG image
        //
        String svgXml =
            WorkflowSvgPainter.generateWorkflowSvg(workflow.getWorkflowMeta(), 1.0f, variables);
        svgStream = new ByteArrayOutputStream();
        try {
          svgStream.write(svgXml.getBytes(StandardCharsets.UTF_8));
        } finally {
          svgStream.flush();
        }
        response.setContentLength(svgStream.size());

        OutputStream out = response.getOutputStream();
        out.write(svgStream.toByteArray());
      }
    } catch (Exception e) {
      e.printStackTrace();
    } finally {
      if (svgStream != null) {
        svgStream.close();
      }
    }
  }

  public String toString() {
    return "Workflow Image Handler";
  }

  @Override
  public String getService() {
    return CONTEXT_PATH + " (" + toString() + ")";
  }

  @Override
  public String getContextPath() {
    return CONTEXT_PATH;
  }
}
