package org.apache.torque.templates.typemapping;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import org.apache.commons.lang.StringUtils;

/**
 * <p>
 * Supports the Derby / DB2 / SQL92 standard for defining Binary data fields
 * with either CHAR(#) FOR BIT DATA or VARCHAR(#) FOR BIT DATA. This can be
 * used in Platform implementors initialize() methods, by using lines like:
 * </p>
 * <code>
 * setSchemaDomainMapping(new SizedForBitDataDomain(
 *                          SchemaType.BINARY, "CHAR", "1"));
 * setSchemaDomainMapping(new SizedForBitDataDomain(
 *                          SchemaType.VARBINARY, "VARCHAR"));
 * </code>
 * <p>
 * This will cause the Column.getSqlString() method to produce items similar to:
 * </p>
 * <code>
 * CHAR(#) FOR BIT DATA
 * VARCHAR(#)FOR BIT DATA
 * </code>
 * <p>
 * Where: # is the size= schema attribute or a default size specified in the
 * constructor.
 * </p>
 * <p>
 * Note that this is dependent on the platform implementation correctly defining
 * BINARY and VARBINARY as having a size attribute in the "hasSize()" method.
 * </p>
 *
 * @see org.apache.torque.templates.platform.Platform
 * @see org.apache.torque.templates.transformer.sql.SQLTransformer#getSqlString()
 * @author <a href="mailto:Monroe@DukeCE.com">Greg Monroe</a>
 */
public class SizedForBitDataSqlType extends SqlType
{
    /**
     * @see SqlType#SqlType(String)
     */
    public SizedForBitDataSqlType(String sqlType)
    {
        super(sqlType);
    }

    /**
     * @see SqlType#SqlType(String, String)
     */
    public SizedForBitDataSqlType(String sqlType, String size)
    {
        super(sqlType, size);
    }

    /**
     * @see SqlType#SqlType(String, String, string, String)
     */
    public SizedForBitDataSqlType(
            SqlType sqlType,
            String size,
            String scale,
            String defaultValue)
    {
        super(sqlType, size, scale, defaultValue);
    }

    /**
     * Returns the size postfix for the base SQL Column type.
     *
     * @return "(size) FOR BIT DATA" or just " FOR BIT DATA" if size
     * is null.
     * @see org.apache.torque.SqlType.database.model.Domain#getSize()
     */
    @Override
    public String printSize(String sizeSuffix)
    {
        String result = "";
        if (!StringUtils.isBlank(getSize()))
        {
            result =  "(" + getSize() + sizeSuffix + ")";
        }
        result = result + " FOR BIT DATA";
        return result;
    }

    /**
     * Returns a new instance with the given sqlTypeName, size, scale
     * and default value.
     *
     * @return a new instance with the given parameters.
     */
    @Override
    public SqlType getNew(
            String size,
            String scale,
            String defaultValue)
    {
        if (size == null)
        {
            size = "1";;
        }
        return new SizedForBitDataSqlType(
                this,
                size,
                scale,
                defaultValue);
    }

}
