/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.utils;

import com.codahale.metrics.Meter;
import com.google.common.util.concurrent.SidecarRateLimiter;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.google.inject.name.Named;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import io.vertx.ext.web.handler.HttpException;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.sidecar.common.exceptions.RangeException;
import org.apache.cassandra.sidecar.common.utils.HttpRange;
import org.apache.cassandra.sidecar.concurrent.ExecutorPools;
import org.apache.cassandra.sidecar.config.ServiceConfiguration;
import org.apache.cassandra.sidecar.config.ThrottleConfiguration;
import org.apache.cassandra.sidecar.metrics.DeltaGauge;
import org.apache.cassandra.sidecar.metrics.instance.InstanceMetrics;
import org.apache.cassandra.sidecar.metrics.instance.StreamSSTableMetrics;
import org.apache.cassandra.sidecar.models.HttpResponse;
import org.apache.cassandra.sidecar.utils.HttpExceptions;
import org.apache.cassandra.sidecar.utils.InstanceMetadataFetcher;
import org.apache.cassandra.sidecar.utils.MetricUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class FileStreamer {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileStreamer.class);
    private final ExecutorPools executorPools;
    private final ThrottleConfiguration config;
    private final SidecarRateLimiter rateLimiter;
    private final InstanceMetadataFetcher instanceMetadataFetcher;

    @Inject
    public FileStreamer(ExecutorPools executorPools, ServiceConfiguration config, @Named(value="StreamRequestRateLimiter") SidecarRateLimiter rateLimiter, InstanceMetadataFetcher instanceMetadataFetcher) {
        this.executorPools = executorPools;
        this.config = config.throttleConfiguration();
        this.rateLimiter = rateLimiter;
        this.instanceMetadataFetcher = instanceMetadataFetcher;
    }

    public Future<Void> stream(HttpResponse response, int instanceId, String filename, long fileLength, String rangeHeader) {
        return this.parseRangeHeader(rangeHeader, fileLength).compose(range -> this.stream(response, instanceId, filename, fileLength, (HttpRange)range));
    }

    public Future<Void> stream(HttpResponse response, int instanceId, String filename, long fileLength, HttpRange range) {
        Promise promise = Promise.promise();
        try {
            this.acquireAndSend(response, instanceId, filename, fileLength, range, System.nanoTime(), (Promise<Void>)promise);
        }
        catch (Throwable t) {
            promise.tryFail(t);
        }
        return promise.future();
    }

    private void acquireAndSend(HttpResponse response, int instanceId, String filename, long fileLength, HttpRange range, long startTimeNanos, Promise<Void> promise) {
        InstanceMetrics instanceMetrics = this.instanceMetadataFetcher.instance(instanceId).metrics();
        StreamSSTableMetrics streamSSTableMetrics = instanceMetrics.streamSSTable();
        if (this.acquire(response, instanceId, filename, fileLength, range, startTimeNanos, streamSSTableMetrics, promise)) {
            LOGGER.debug("Streaming range {} for file {} to client {}. Instance: {}", new Object[]{range, filename, response.remoteAddress(), response.host()});
            response.sendFile(filename, fileLength, range).onSuccess(v -> {
                String component = MetricUtils.parseSSTableComponent(filename);
                LOGGER.debug("Streamed file {} successfully to client {}. Instance: {}", new Object[]{filename, response.remoteAddress(), response.host()});
                ((Meter)streamSSTableMetrics.forComponent((String)component).bytesStreamedRate.metric).mark(range.length());
                ((Meter)instanceMetrics.streamSSTable().totalBytesStreamedRate.metric).mark(range.length());
                promise.complete();
            }).onFailure(arg_0 -> promise.fail(arg_0));
        }
    }

    private boolean acquire(HttpResponse response, int instanceId, String filename, long fileLength, HttpRange range, long startTimeNanos, StreamSSTableMetrics streamSSTableMetrics, Promise<Void> promise) {
        if (this.rateLimiter.tryAcquire()) {
            return true;
        }
        long waitTimeNanos = TimeUnit.MICROSECONDS.toNanos(this.rateLimiter.queryWaitTimeInMicros());
        if (this.isTimeoutExceeded(startTimeNanos, waitTimeNanos)) {
            LOGGER.warn("Retries for acquiring permit exhausted for client {}. Instance: {}. Asking client to retry after {} nanoseconds.", new Object[]{response.remoteAddress(), response.host(), waitTimeNanos});
            response.setRetryAfterHeader(waitTimeNanos);
            ((DeltaGauge)streamSSTableMetrics.throttled.metric).update(1L);
            promise.fail((Throwable)new HttpException(HttpResponseStatus.TOO_MANY_REQUESTS.code(), "Retry exhausted"));
        } else {
            LOGGER.debug("Retrying streaming after {} nanos for client {}. Instance: {}", new Object[]{waitTimeNanos, response.remoteAddress(), response.host()});
            this.executorPools.service().setTimer(TimeUnit.NANOSECONDS.toMillis(waitTimeNanos) + 1L, (Handler<Long>)((Handler)t -> this.acquireAndSend(response, instanceId, filename, fileLength, range, startTimeNanos, promise)));
        }
        return false;
    }

    private boolean isTimeoutExceeded(long startTimeNanos, long waitTimeNanos) {
        long nowNanos = System.nanoTime();
        long timeoutNanos = startTimeNanos + waitTimeNanos + this.config.timeout().to(TimeUnit.NANOSECONDS);
        return timeoutNanos < nowNanos;
    }

    public Future<HttpRange> parseRangeHeader(String rangeHeader, long fileLength) {
        HttpRange fr = HttpRange.of((long)0L, (long)(fileLength - 1L));
        if (rangeHeader == null) {
            return Future.succeededFuture((Object)fr);
        }
        try {
            HttpRange hr = HttpRange.parseHeader((String)rangeHeader, (long)fileLength);
            HttpRange intersect = fr.intersect(hr);
            LOGGER.debug("Calculated range {} for streaming", (Object)intersect);
            return Future.succeededFuture((Object)intersect);
        }
        catch (IllegalArgumentException | UnsupportedOperationException | RangeException e) {
            LOGGER.error("Failed to parse header '{}'", (Object)rangeHeader, (Object)e);
            return Future.failedFuture((Throwable)HttpExceptions.wrapHttpException(HttpResponseStatus.REQUESTED_RANGE_NOT_SATISFIABLE, e.getMessage(), e));
        }
    }
}

